/*
** Zabbix
** Copyright (C) 2001-2024 Zabbix SIA
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

package conn

import (
	"net"
	"testing"
	"time"

	"github.com/google/go-cmp/cmp"
	"golang.zabbix.com/plugin/ember-plus/ember"
	"golang.zabbix.com/sdk/errs"
	"golang.zabbix.com/sdk/log"
)

func Test_connHandler_read(t *testing.T) {
	t.Parallel()

	type args struct {
		message  [][]byte
		deadline int
	}

	tests := []struct {
		name    string
		args    args
		want    []byte
		wantErr bool
	}{
		{
			"+valid",
			args{
				[][]byte{
					{
						0x02, 0x00, 0x00, 0x00, 0x45, 0x00, 0x00, 0x6c, 0x34, 0x95, 0x40, 0x00, 0x80, 0x06, 0x00, 0x00,
						0x7f, 0x00, 0x00, 0x01, 0x7f, 0x00, 0x00, 0x01, 0x27, 0x0e, 0xd8, 0x4f, 0x5f, 0x85, 0xe1, 0x0e,
						0x46, 0x58, 0x3e, 0xfd, 0x50, 0x18, 0x27, 0xf9, 0x11, 0xbf, 0x00, 0x00, 0xfe, 0x00, 0x0e, 0x00,
						0x01, 0xc0, 0x01, 0x02, 0x1e, 0x02, 0x60, 0x34, 0x6b, 0x32, 0xa0, 0x30, 0x63, 0x2e, 0xa0, 0x03,
						0x02, 0x01, 0x01, 0xa1, 0x27, 0x31, 0x25, 0xa0, 0x16, 0x0c, 0x14, 0x52, 0x33, 0x4c, 0x41, 0x59,
						0x56, 0x69, 0x72, 0x74, 0x75, 0x61, 0x6c, 0x50, 0x61, 0x74, 0x63, 0x68, 0x42, 0x61, 0x79, 0xa1,
						0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xb5, 0x11, 0xff,
					},
				},
				3,
			},
			[]byte{
				0x60, 0x34, 0x6b, 0x32, 0xa0, 0x30, 0x63, 0x2e, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x27, 0x31, 0x25,
				0xa0, 0x16, 0x0c, 0x14, 0x52, 0x33, 0x4c, 0x41, 0x59, 0x56, 0x69, 0x72, 0x74, 0x75, 0x61, 0x6c, 0x50,
				0x61, 0x74, 0x63, 0x68, 0x42, 0x61, 0x79, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03,
				0x01, 0x01, 0xff,
			},
			false,
		},
		{
			"+zeroStart",
			args{
				[][]byte{
					{
						0x00, 0x50, 0x56, 0x8f, 0xbe, 0x39, 0xf8, 0x0f, 0x6f, 0xcb, 0xce, 0x93, 0x08, 0x00, 0x45, 0x00,
						0x00, 0x6c, 0x00, 0x00, 0x40, 0x00, 0x3e, 0x06, 0x4e, 0x11, 0xac, 0x1e, 0x48, 0x0a, 0xac, 0x1e,
						0x4e, 0x34, 0x23, 0x29, 0xb6, 0xb6, 0xab, 0x83, 0x49, 0x52, 0x66, 0x29, 0x59, 0x3d, 0x80, 0x18,
						0x00, 0x80, 0x78, 0x0d, 0x00, 0x00, 0x01, 0x01, 0x08, 0x0a, 0x47, 0xa2, 0x67, 0x27, 0x03, 0xc4,
						0xbe, 0x0a, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0xc0, 0x01, 0x02, 0x14, 0x02, 0x60, 0x80, 0x6b, 0x80,
						0xa0, 0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c,
						0x04, 0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x9b, 0xff,
					},
				},
				3,
			},
			[]byte{
				0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80,
				0xa0, 0x06, 0x0c, 0x04, 0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			false,
		},
		{
			"+multi",
			args{
				[][]byte{
					{
						0x0a, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0x80, 0x01, 0x02, 0x14, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0,
						0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04,
						0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x9b, 0xff,
					},
					{
						0x0a, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0x40, 0x01, 0x02, 0x14, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0,
						0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04,
						0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x9b, 0xff,
					},
				},
				3,
			},
			[]byte{
				0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80,
				0xa0, 0x06, 0x0c, 0x04, 0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x63, 0x80, 0xa0,
				0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04, 0x52, 0x75, 0x62, 0x79, 0xa3,
				0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			false,
		},
		{
			"+multiWithAdditionalPackets",
			args{
				[][]byte{
					{
						0x0a, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0x80, 0x01, 0x02, 0x14, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0,
						0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04,
						0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x9b, 0xff,
					},
					{
						0x0a, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0x00, 0x01, 0x02, 0x14, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0,
						0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04,
						0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x9b, 0xff},
					{
						0x0a, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0x40, 0x01, 0x02, 0x14, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0,
						0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04,
						0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x9b, 0xff,
					},
				},
				3,
			},
			[]byte{
				0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80,
				0xa0, 0x06, 0x0c, 0x04, 0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x63, 0x80, 0xa0,
				0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04, 0x52, 0x75, 0x62, 0x79, 0xa3,
				0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60,
				0x80, 0x6b, 0x80, 0xa0, 0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0,
				0x06, 0x0c, 0x04, 0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			false,
		},
		{
			"+multiWithAdditionalSkipPacket",
			args{
				[][]byte{
					{
						0x0a, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0x80, 0x01, 0x02, 0x14, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0,
						0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04,
						0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x9b, 0xff,
					},
					{
						0x0a, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0x00, 0x01, 0x02, 0x14, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0,
						0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04,
						0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x9b, 0xff},
					{
						0x0a, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0xC0, 0x01, 0x02, 0x14, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0,
						0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04,
						0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x9b, 0xff,
					},
					{
						0x0a, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0x40, 0x01, 0x02, 0x14, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0,
						0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04,
						0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x9b, 0xff,
					},
				},
				3,
			},
			[]byte{
				0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80,
				0xa0, 0x06, 0x0c, 0x04, 0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x63, 0x80, 0xa0,
				0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04, 0x52, 0x75, 0x62, 0x79, 0xa3,
				0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60,
				0x80, 0x6b, 0x80, 0xa0, 0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0,
				0x06, 0x0c, 0x04, 0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			false,
		},
		{
			"+multiInSingle",
			args{
				[][]byte{
					{
						0x0a, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0x80, 0x01, 0x02, 0x14, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0,
						0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04,
						0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x9b, 0xff, 0x0a, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0x40,
						0x01, 0x02, 0x14, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01,
						0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04, 0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01,
						0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f,
						0x9b, 0xff,
					},
				},
				3,
			},
			[]byte{
				0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80,
				0xa0, 0x06, 0x0c, 0x04, 0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x63, 0x80, 0xa0,
				0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04, 0x52, 0x75, 0x62, 0x79, 0xa3,
				0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			false,
		},
		{
			"+multiMix",
			args{
				[][]byte{
					{
						0x0a, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0x80, 0x01, 0x02, 0x14, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0,
						0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04,
						0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x9b, 0xff, 0x0a, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0x00,
						0x01, 0x02, 0x14, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01,
						0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04, 0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01,
						0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f,
						0x9b, 0xff,
					},
					{
						0x0a, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0x40, 0x01, 0x02, 0x14, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0,
						0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04,
						0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x9b, 0xff,
					},
				},
				3,
			},
			[]byte{
				0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80,
				0xa0, 0x06, 0x0c, 0x04, 0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x63, 0x80, 0xa0,
				0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04, 0x52, 0x75, 0x62, 0x79, 0xa3,
				0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60,
				0x80, 0x6b, 0x80, 0xa0, 0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0,
				0x06, 0x0c, 0x04, 0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			false,
		},
		{
			"+unfinishedIncompletePacketWithValidAfter",
			args{
				[][]byte{
					{
						0x02, 0x00, 0x00, 0x00, 0x45, 0x00, 0x00, 0x6c, 0x34, 0x95, 0x40, 0x00, 0x80, 0x06, 0x00, 0x00,
						0x7f, 0x00, 0x00, 0x01, 0x7f, 0x00, 0x00, 0x01, 0x27, 0x0e, 0xd8, 0x4f, 0x5f, 0x85, 0xe1, 0x0e,
						0x46, 0x58, 0x3e, 0xfd, 0x50, 0x18, 0x27, 0xf9, 0x11, 0xbf, 0x00, 0x00, 0xfe, 0x00, 0x0e, 0x00,
						0x01, 0xc0, 0x01, 0x02, 0x1e, 0x02, 0x60, 0x34, 0x6b, 0x32, 0xa0,
					},
					{
						0x02, 0x00, 0x00, 0x00, 0x45, 0x00, 0x00, 0x6c, 0x34, 0x95, 0x40, 0x00, 0x80, 0x06, 0x00, 0x00,
						0x7f, 0x00, 0x00, 0x01, 0x7f, 0x00, 0x00, 0x01, 0x27, 0x0e, 0xd8, 0x4f, 0x5f, 0x85, 0xe1, 0x0e,
						0x46, 0x58, 0x3e, 0xfd, 0x50, 0x18, 0x27, 0xf9, 0x11, 0xbf, 0x00, 0x00, 0xfe, 0x00, 0x0e, 0x00,
						0x01, 0xc0, 0x01, 0x02, 0x1e, 0x02, 0x60, 0x34, 0x6b, 0x32, 0xa0, 0x30, 0x63, 0x2e, 0xa0, 0x03,
						0x02, 0x01, 0x01, 0xa1, 0x27, 0x31, 0x25, 0xa0, 0x16, 0x0c, 0x14, 0x52, 0x33, 0x4c, 0x41, 0x59,
						0x56, 0x69, 0x72, 0x74, 0x75, 0x61, 0x6c, 0x50, 0x61, 0x74, 0x63, 0x68, 0x42, 0x61, 0x79, 0xa1,
						0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xb5, 0x11, 0xff,
					},
				},
				3,
			},
			[]byte{
				0x60, 0x34, 0x6b, 0x32, 0xa0, 0x30, 0x63, 0x2e, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x27, 0x31, 0x25,
				0xa0, 0x16, 0x0c, 0x14, 0x52, 0x33, 0x4c, 0x41, 0x59, 0x56, 0x69, 0x72, 0x74, 0x75, 0x61, 0x6c, 0x50,
				0x61, 0x74, 0x63, 0x68, 0x42, 0x61, 0x79, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03,
				0x01, 0x01, 0xff,
			},
			false,
		},
		{
			"+singleS101MultiPacket",
			args{
				[][]byte{
					{
						0x00, 0x50, 0x56, 0x8f, 0x12, 0x70, 0xe0, 0x69, 0xba, 0x8f, 0x03, 0x87, 0x08, 0x00, 0x45, 0x00,
						0x05, 0xdc, 0xdc, 0xf2, 0x40, 0x00, 0x7e, 0x06, 0x29, 0x80, 0xac, 0x1e, 0x4f, 0xa3, 0xac, 0x1e,
						0x48, 0xc9, 0x25, 0x1d, 0xce, 0xeb, 0xf2, 0xd5, 0xd7, 0x2a, 0xee, 0xde, 0xf7, 0x91, 0x50, 0x10,
						0x20, 0x13, 0x25, 0xff, 0x00, 0x00, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0xc0, 0x01, 0x02, 0x1e, 0x02,
						0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x6a, 0x80, 0xa0, 0x06, 0x0d, 0x04, 0x01, 0x03, 0x02, 0x04,
						0xa1, 0x80, 0x31, 0x80, 0xa0, 0x08, 0x0c, 0x06, 0x53, 0x74, 0x72, 0x69, 0x6e, 0x67, 0xa2, 0x03,
						0x01, 0x01, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x01, 0xa1, 0x80,
						0x31, 0x80, 0xa0, 0x08, 0x0c, 0x06, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x31, 0xa2, 0x0a, 0x0c, 0x08,
						0x53, 0x74, 0x75, 0x64, 0x69, 0x6f, 0x20, 0x41, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01,
						0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x02, 0xa1, 0x80, 0x31,
						0x80, 0xa0, 0x08, 0x0c, 0x06, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x32, 0xa2, 0x0a, 0x0c, 0x08, 0x4d,
						0x30, 0x33, 0x20, 0x20, 0x20, 0x20, 0x20, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01,
						0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0,
						0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x03, 0xa1, 0x80, 0x31, 0x80,
						0xa0, 0x08, 0x0c, 0x06, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x33, 0xa2, 0x06, 0x0c, 0x04, 0x53, 0x6f,
						0x6c, 0x6f, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02,
						0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07,
						0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x04, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x08, 0x0c, 0x06, 0x67,
						0x70, 0x69, 0x6f, 0x2d, 0x34, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03,
						0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x05, 0xa1, 0x80,
						0x31, 0x80, 0xa0, 0x08, 0x0c, 0x06, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x35, 0xa2, 0x02, 0x0c, 0x00,
						0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05,
						0x01, 0x03, 0x02, 0x04, 0x06, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x08, 0x0c, 0x06, 0x67, 0x70, 0x69,
						0x6f, 0x2d, 0x36, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01,
						0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0,
						0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x07, 0xa1, 0x80, 0x31, 0x80,
						0xa0, 0x08, 0x0c, 0x06, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x37, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03,
						0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
						0x02, 0x04, 0x08, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x08, 0x0c, 0x06, 0x67, 0x70, 0x69, 0x6f, 0x2d,
						0x38, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf,
						0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69,
						0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x09, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x08,
						0x0c, 0x06, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x39, 0xa2, 0x08, 0x0c, 0x06, 0x55, 0x73, 0x65, 0x72,
						0x30, 0x31, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02,
						0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07,
						0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x0a, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67,
						0x70, 0x69, 0x6f, 0x2d, 0x31, 0x30, 0xa2, 0x08, 0x0c, 0x06, 0x52, 0x45, 0x4d, 0x4f, 0x54, 0x45,
						0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05,
						0x01, 0x03, 0x02, 0x04, 0x0b, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69,
						0x6f, 0x2d, 0x31, 0x31, 0xa2, 0x2a, 0x0c, 0x28, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x6e,
						0x79, 0x2d, 0x61, 0x6c, 0x61, 0x72, 0x6d, 0x6c, 0x6f, 0x67, 0x30, 0x31, 0x2f, 0x6c, 0x73, 0x63,
						0x6c, 0x69, 0x65, 0x6e, 0x74, 0x2f, 0x75, 0x73, 0x65, 0x72, 0x30, 0x31, 0x2e, 0x68, 0x74, 0x6d,
						0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05,
						0x01, 0x03, 0x02, 0x04, 0x0c, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69,
						0x6f, 0x2d, 0x31, 0x32, 0xa2, 0x0f, 0x0c, 0x0d, 0x47, 0x65, 0x6e, 0x65, 0x20, 0x47, 0x65, 0x72,
						0x68, 0x69, 0x73, 0x65, 0x72, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf,
						0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69,
						0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x0d, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09,
						0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x31, 0x33, 0xa2, 0x09, 0x0c, 0x07, 0x53, 0x54, 0x41,
						0x20, 0x4d, 0x49, 0x43, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad,
						0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80,
						0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x0e, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c,
						0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x31, 0x34, 0xa2, 0x09, 0x0c, 0x07, 0x53, 0x54, 0x41, 0x20,
						0x49, 0x4e, 0x54, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03,
						0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0,
						0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x0f, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07,
						0x67, 0x70, 0x69, 0x6f, 0x2d, 0x31, 0x35, 0xa2, 0x0f, 0x0c, 0x0d, 0x31, 0x37, 0x32, 0x2e, 0x33,
						0x30, 0x2e, 0x37, 0x34, 0x2e, 0x31, 0x30, 0x31, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01,
						0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x10, 0xa1, 0x80, 0x31,
						0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x31, 0x36, 0xa2, 0x07, 0x0c, 0x05,
						0x31, 0x39, 0x36, 0x30, 0x31, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf,
						0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69,
						0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x11, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09,
						0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x31, 0x37, 0xa2, 0x0f, 0x0c, 0x0d, 0x31, 0x37, 0x32,
						0x2e, 0x33, 0x30, 0x2e, 0x37, 0x39, 0x2e, 0x31, 0x36, 0x33, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9,
						0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x12, 0xa1,
						0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x31, 0x38, 0xa2, 0x07,
						0x0c, 0x05, 0x31, 0x32, 0x35, 0x36, 0x33, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01,
						0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0,
						0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x13, 0xa1, 0x80, 0x31, 0x80,
						0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x31, 0x39, 0xa2, 0x07, 0x0c, 0x05, 0x33,
						0x30, 0x30, 0x30, 0x33, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad,
						0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80,
						0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x14, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c,
						0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x32, 0x30, 0xa2, 0x0e, 0x0c, 0x0c, 0x31, 0x37, 0x32, 0x2e,
						0x33, 0x30, 0x2e, 0x37, 0x39, 0x2e, 0x37, 0x31, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01,
						0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x15, 0xa1, 0x80, 0x31,
						0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x32, 0x31, 0xa2, 0x02, 0x0c, 0x00,
						0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05,
						0x01, 0x03, 0x02, 0x04, 0x16, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69,
						0x6f, 0x2d, 0x32, 0x32, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01,
						0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x17, 0xa1, 0x80, 0x31,
						0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d,
					},
					{
						0x32, 0x33, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd,
						0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80,
						0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x18, 0xa1, 0x80, 0x31, 0x80, 0xa0,
						0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x32, 0x34, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03,
						0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
						0x02, 0x04, 0x19, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d,
						0x32, 0x35, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd,
						0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80,
						0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x1a, 0xa1, 0x80, 0x31, 0x80, 0xa0,
						0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x32, 0x36, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03,
						0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
						0x02, 0x04, 0x1b, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d,
						0x32, 0x37, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd,
						0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80,
						0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x1c, 0xa1, 0x80, 0x31, 0x80, 0xa0,
						0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x32, 0x38, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03,
						0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
						0x02, 0x04, 0x1d, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d,
						0x32, 0x39, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd,
						0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80,
						0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x1e, 0xa1, 0x80, 0x31, 0x80, 0xa0,
						0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x33, 0x30, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03,
						0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
						0x02, 0x04, 0x1f, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d,
						0x33, 0x31, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd,
						0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80,
						0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x20, 0xa1, 0x80, 0x31, 0x80, 0xa0,
						0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x33, 0x32, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03,
						0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
						0x02, 0x04, 0x21, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d,
						0x33, 0x33, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd,
						0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80,
						0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x22, 0xa1, 0x80, 0x31, 0x80, 0xa0,
						0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x33, 0x34, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03,
						0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
						0x02, 0x04, 0x23, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d,
						0x33, 0x35, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd,
						0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80,
						0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x24, 0xa1, 0x80, 0x31, 0x80, 0xa0,
						0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x33, 0x36, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03,
						0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
						0x02, 0x04, 0x25, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d,
						0x33, 0x37, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd,
						0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80,
						0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x26, 0xa1, 0x80, 0x31, 0x80, 0xa0,
						0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x33, 0x38, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03,
						0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
						0x02, 0x04, 0x27, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d,
						0x33, 0x39, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd,
						0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80,
						0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x28, 0xa1, 0x80, 0x31, 0x80, 0xa0,
						0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x34, 0x30, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03,
						0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
						0x02, 0x04, 0x29, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d,
						0x34, 0x31, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd,
						0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80,
						0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x2a, 0xa1, 0x80, 0x31, 0x80, 0xa0,
						0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x34, 0x32, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03,
						0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
						0x02, 0x04, 0x2b, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d,
						0x34, 0x33, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd,
						0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80,
						0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x2c, 0xa1, 0x80, 0x31, 0x80, 0xa0,
						0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x34, 0x34, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03,
						0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
						0x02, 0x04, 0x2d, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d,
						0x34, 0x35, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd,
						0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80,
						0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x2e, 0xa1, 0x80, 0x31, 0x80, 0xa0,
						0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x34, 0x36, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03,
						0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
						0x02, 0x04, 0x2f, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d,
						0x34, 0x37, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd,
						0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80,
						0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x30, 0xa1, 0x80, 0x31, 0x80, 0xa0,
						0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x34, 0x38, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03,
						0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
						0x02, 0x04, 0x31, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d,
						0x34, 0x39, 0xa2, 0x02,
					},
					{
						0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02,
						0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07,
						0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x32, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67,
						0x70, 0x69, 0x6f, 0x2d, 0x35, 0x30, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9,
						0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x33, 0xa1,
						0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x35, 0x31, 0xa2, 0x02,
						0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02,
						0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07,
						0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x34, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67,
						0x70, 0x69, 0x6f, 0x2d, 0x35, 0x32, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9,
						0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x35, 0xa1,
						0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x35, 0x33, 0xa2, 0x02,
						0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02,
						0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07,
						0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x36, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67,
						0x70, 0x69, 0x6f, 0x2d, 0x35, 0x34, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9,
						0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x37, 0xa1,
						0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x35, 0x35, 0xa2, 0x02,
						0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02,
						0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07,
						0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x38, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67,
						0x70, 0x69, 0x6f, 0x2d, 0x35, 0x36, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9,
						0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x39, 0xa1,
						0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x35, 0x37, 0xa2, 0x02,
						0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02,
						0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07,
						0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x3a, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67,
						0x70, 0x69, 0x6f, 0x2d, 0x35, 0x38, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9,
						0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x3b, 0xa1,
						0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x35, 0x39, 0xa2, 0x02,
						0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02,
						0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07,
						0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x3c, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67,
						0x70, 0x69, 0x6f, 0x2d, 0x36, 0x30, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9,
						0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x3d, 0xa1,
						0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x36, 0x31, 0xa2, 0x02,
						0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02,
						0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07,
						0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x3e, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67,
						0x70, 0x69, 0x6f, 0x2d, 0x36, 0x32, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9,
						0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x3f, 0xa1,
						0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x36, 0x33, 0xa2, 0x02,
						0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02,
						0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07,
						0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x40, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67,
						0x70, 0x69, 0x6f, 0x2d, 0x36, 0x34, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9,
						0x03, 0x01, 0x01, 0xfd, 0xdf, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1b, 0xa3, 0xff,
					},
				},
				4,
			},
			[]byte{
				0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x6a, 0x80, 0xa0, 0x06, 0x0d, 0x04, 0x01, 0x03, 0x02, 0x04, 0xa1,
				0x80, 0x31, 0x80, 0xa0, 0x08, 0x0c, 0x06, 0x53, 0x74, 0x72, 0x69, 0x6e, 0x67, 0xa2, 0x03, 0x01, 0x01,
				0x00, 0xa3, 0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69,
				0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x08, 0x0c,
				0x06, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x31, 0xa2, 0x0a, 0x0c, 0x08, 0x53, 0x74, 0x75, 0x64, 0x69, 0x6f,
				0x20, 0x41, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01,
				0x03, 0x02, 0x04, 0x02, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x08, 0x0c, 0x06, 0x67, 0x70, 0x69, 0x6f, 0x2d,
				0x32, 0xa2, 0x0a, 0x0c, 0x08, 0x4d, 0x30, 0x33, 0x20, 0x20, 0x20, 0x20, 0x20, 0xa5, 0x03, 0x02, 0x01,
				0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x03, 0xa1, 0x80,
				0x31, 0x80, 0xa0, 0x08, 0x0c, 0x06, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x33, 0xa2, 0x06, 0x0c, 0x04, 0x53,
				0x6f, 0x6c, 0x6f, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01,
				0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05,
				0x01, 0x03, 0x02, 0x04, 0x04, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x08, 0x0c, 0x06, 0x67, 0x70, 0x69, 0x6f,
				0x2d, 0x34, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad,
				0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0,
				0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x05, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x08, 0x0c, 0x06, 0x67,
				0x70, 0x69, 0x6f, 0x2d, 0x35, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01,
				0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80,
				0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x06, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x08,
				0x0c, 0x06, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x36, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03,
				0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x07, 0xa1, 0x80, 0x31,
				0x80, 0xa0, 0x08, 0x0c, 0x06, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x37, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03,
				0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x08,
				0xa1, 0x80, 0x31, 0x80, 0xa0, 0x08, 0x0c, 0x06, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x38, 0xa2, 0x02, 0x0c,
				0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
				0x02, 0x04, 0x09, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x08, 0x0c, 0x06, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x39,
				0xa2, 0x08, 0x0c, 0x06, 0x55, 0x73, 0x65, 0x72, 0x30, 0x31, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03,
				0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0,
				0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x0a, 0xa1, 0x80, 0x31, 0x80, 0xa0,
				0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x31, 0x30, 0xa2, 0x08, 0x0c, 0x06, 0x52, 0x45, 0x4d,
				0x4f, 0x54, 0x45, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01,
				0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05,
				0x01, 0x03, 0x02, 0x04, 0x0b, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f,
				0x2d, 0x31, 0x31, 0xa2, 0x2a, 0x0c, 0x28, 0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x6e, 0x79, 0x2d,
				0x61, 0x6c, 0x61, 0x72, 0x6d, 0x6c, 0x6f, 0x67, 0x30, 0x31, 0x2f, 0x6c, 0x73, 0x63, 0x6c, 0x69, 0x65,
				0x6e, 0x74, 0x2f, 0x75, 0x73, 0x65, 0x72, 0x30, 0x31, 0x2e, 0x68, 0x74, 0x6d, 0xa5, 0x03, 0x02, 0x01,
				0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x0c, 0xa1, 0x80,
				0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x31, 0x32, 0xa2, 0x0f, 0x0c, 0x0d,
				0x47, 0x65, 0x6e, 0x65, 0x20, 0x47, 0x65, 0x72, 0x68, 0x69, 0x73, 0x65, 0x72, 0xa5, 0x03, 0x02, 0x01,
				0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x0d, 0xa1, 0x80,
				0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x31, 0x33, 0xa2, 0x09, 0x0c, 0x07,
				0x53, 0x54, 0x41, 0x20, 0x4d, 0x49, 0x43, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff,
				0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80,
				0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x0e, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07,
				0x67, 0x70, 0x69, 0x6f, 0x2d, 0x31, 0x34, 0xa2, 0x09, 0x0c, 0x07, 0x53, 0x54, 0x41, 0x20, 0x49, 0x4e,
				0x54, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
				0x02, 0x04, 0x0f, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x31,
				0x35, 0xa2, 0x0f, 0x0c, 0x0d, 0x31, 0x37, 0x32, 0x2e, 0x33, 0x30, 0x2e, 0x37, 0x34, 0x2e, 0x31, 0x30,
				0x31, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
				0x02, 0x04, 0x10, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x31,
				0x36, 0xa2, 0x07, 0x0c, 0x05, 0x31, 0x39, 0x36, 0x30, 0x31, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03,
				0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0,
				0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x11, 0xa1, 0x80, 0x31, 0x80, 0xa0,
				0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x31, 0x37, 0xa2, 0x0f, 0x0c, 0x0d, 0x31, 0x37, 0x32,
				0x2e, 0x33, 0x30, 0x2e, 0x37, 0x39, 0x2e, 0x31, 0x36, 0x33, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03,
				0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0,
				0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x12, 0xa1, 0x80, 0x31, 0x80, 0xa0,
				0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x31, 0x38, 0xa2, 0x07, 0x0c, 0x05, 0x31, 0x32, 0x35,
				0x36, 0x33, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01,
				0x03, 0x02, 0x04, 0x13, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d,
				0x31, 0x39, 0xa2, 0x07, 0x0c, 0x05, 0x33, 0x30, 0x30, 0x30, 0x33, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9,
				0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x14, 0xa1, 0x80, 0x31, 0x80,
				0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x32, 0x30, 0xa2, 0x0e, 0x0c, 0x0c, 0x31, 0x37,
				0x32, 0x2e, 0x33, 0x30, 0x2e, 0x37, 0x39, 0x2e, 0x37, 0x31, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03,
				0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0,
				0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x15, 0xa1, 0x80, 0x31, 0x80, 0xa0,
				0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x32, 0x31, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02,
				0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x16, 0xa1,
				0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x32, 0x32, 0xa2, 0x02, 0x0c,
				0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
				0x02, 0x04, 0x17, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x32,
				0x33, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03,
				0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07,
				0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x18, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70,
				0x69, 0x6f, 0x2d, 0x32, 0x34, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01,
				0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80,
				0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x19, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09,
				0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x32, 0x35, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01,
				0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x1a, 0xa1, 0x80,
				0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x32, 0x36, 0xa2, 0x02, 0x0c, 0x00,
				0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02,
				0x04, 0x1b, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x32, 0x37,
				0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02,
				0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d,
				0x05, 0x01, 0x03, 0x02, 0x04, 0x1c, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69,
				0x6f, 0x2d, 0x32, 0x38, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01,
				0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69,
				0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x1d, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c,
				0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x32, 0x39, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03,
				0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x1e, 0xa1, 0x80, 0x31,
				0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x33, 0x30, 0xa2, 0x02, 0x0c, 0x00, 0xa5,
				0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04,
				0x1f, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x33, 0x31, 0xa2,
				0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01,
				0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05,
				0x01, 0x03, 0x02, 0x04, 0x20, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f,
				0x2d, 0x33, 0x32, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff,
				0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80,
				0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x21, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07,
				0x67, 0x70, 0x69, 0x6f, 0x2d, 0x33, 0x33, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9,
				0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x22, 0xa1, 0x80, 0x31, 0x80,
				0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x33, 0x34, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03,
				0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x23,
				0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x33, 0x35, 0xa2, 0x02,
				0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01,
				0x03, 0x02, 0x04, 0x24, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d,
				0x33, 0x36, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad,
				0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0,
				0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x25, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67,
				0x70, 0x69, 0x6f, 0x2d, 0x33, 0x37, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03,
				0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0,
				0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x26, 0xa1, 0x80, 0x31, 0x80, 0xa0,
				0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x33, 0x38, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02,
				0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x27, 0xa1,
				0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x33, 0x39, 0xa2, 0x02, 0x0c,
				0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
				0x02, 0x04, 0x28, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x34,
				0x30, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03,
				0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07,
				0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x29, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70,
				0x69, 0x6f, 0x2d, 0x34, 0x31, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01,
				0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80,
				0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x2a, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09,
				0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x34, 0x32, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01,
				0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x2b, 0xa1, 0x80,
				0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x34, 0x33, 0xa2, 0x02, 0x0c, 0x00,
				0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02,
				0x04, 0x2c, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x34, 0x34,
				0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02,
				0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d,
				0x05, 0x01, 0x03, 0x02, 0x04, 0x2d, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69,
				0x6f, 0x2d, 0x34, 0x35, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01,
				0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69,
				0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x2e, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c,
				0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x34, 0x36, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03,
				0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x2f, 0xa1, 0x80, 0x31,
				0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x34, 0x37, 0xa2, 0x02, 0x0c, 0x00, 0xa5,
				0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04,
				0x30, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x34, 0x38, 0xa2,
				0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01,
				0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05,
				0x01, 0x03, 0x02, 0x04, 0x31, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f,
				0x2d, 0x34, 0x39, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff,
				0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80,
				0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x32, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07,
				0x67, 0x70, 0x69, 0x6f, 0x2d, 0x35, 0x30, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9,
				0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x33, 0xa1, 0x80, 0x31, 0x80,
				0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x35, 0x31, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03,
				0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x34,
				0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x35, 0x32, 0xa2, 0x02,
				0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01,
				0x03, 0x02, 0x04, 0x35, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d,
				0x35, 0x33, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad,
				0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0,
				0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x36, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67,
				0x70, 0x69, 0x6f, 0x2d, 0x35, 0x34, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03,
				0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0,
				0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x37, 0xa1, 0x80, 0x31, 0x80, 0xa0,
				0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x35, 0x35, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02,
				0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x38, 0xa1,
				0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x35, 0x36, 0xa2, 0x02, 0x0c,
				0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03,
				0x02, 0x04, 0x39, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x35,
				0x37, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03,
				0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07,
				0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x3a, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70,
				0x69, 0x6f, 0x2d, 0x35, 0x38, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01,
				0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80,
				0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x3b, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09,
				0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x35, 0x39, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01,
				0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x3c, 0xa1, 0x80,
				0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x36, 0x30, 0xa2, 0x02, 0x0c, 0x00,
				0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02,
				0x04, 0x3d, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x36, 0x31,
				0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02,
				0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d,
				0x05, 0x01, 0x03, 0x02, 0x04, 0x3e, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69,
				0x6f, 0x2d, 0x36, 0x32, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01,
				0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x69,
				0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x3f, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c,
				0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x36, 0x33, 0xa2, 0x02, 0x0c, 0x00, 0xa5, 0x03, 0x02, 0x01, 0x03,
				0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x03, 0x02, 0x04, 0x40, 0xa1, 0x80, 0x31,
				0x80, 0xa0, 0x09, 0x0c, 0x07, 0x67, 0x70, 0x69, 0x6f, 0x2d, 0x36, 0x34, 0xa2, 0x02, 0x0c, 0x00, 0xa5,
				0x03, 0x02, 0x01, 0x03, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xad, 0x03, 0x02, 0x01, 0x03, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			false,
		},
		{
			"-invalid_packet",
			args{
				[][]byte{
					{
						0x0a, 0x00, 0x00, 0x0e, 0x00, 0x01, 0x40, 0x01, 0x02, 0x14, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0,
						0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04,
						0x52, 0x75, 0x62, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x9b, 0xff,
					},
				},
				1,
			},
			nil,
			true,
		},
		{
			"-empty",
			args{
				[][]byte{},
				1,
			},
			nil,
			true,
		},
	}
	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			server, client := net.Pipe()

			defer func() {
				//nolint:gosec
				server.Close()
				//nolint:gosec
				client.Close()
			}()

			ch := &connHandler{conn: client, logr: log.New("Test")}
			go func() {
				for _, m := range tt.args.message {
					_, err := server.Write(m)
					if err != nil {
						panic(err)
					}
				}
			}()

			err := ch.conn.SetReadDeadline(time.Now().Add(time.Duration(tt.args.deadline) * time.Second))
			if err != nil {
				panic(err)
			}

			got, err := ch.read()
			if (err != nil) != tt.wantErr {
				t.Fatalf("connHandler.read() error = %v, wantErr %v", err, tt.wantErr)
			}

			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("connHandler.read() = %s", diff)
			}
		})
	}
}

func Test_connHandler_readExpected(t *testing.T) {
	t.Parallel()

	type args struct {
		path    string
		timeout int
		message []readResponse
	}

	tests := []struct {
		name    string
		args    args
		want    ember.ElementCollection
		wantErr bool
	}{
		{
			"+valid",
			args{
				"1",
				3,
				[]readResponse{
					{
						data: []byte{
							0x60, 0x34, 0x6b, 0x32, 0xa0, 0x30, 0x63, 0x2e, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x27,
							0x31, 0x25, 0xa0, 0x16, 0x0c, 0x14, 0x52, 0x33, 0x4c, 0x41, 0x59, 0x56, 0x69, 0x72, 0x74,
							0x75, 0x61, 0x6c, 0x50, 0x61, 0x74, 0x63, 0x68, 0x42, 0x61, 0x79, 0xa1, 0x02, 0x0c, 0x00,
							0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xff,
						},
						err: nil,
					},
				},
			},
			ember.ElementCollection{
				ember.ElementKey{
					ID: "R3LAYVirtualPatchBay", Path: "1",
				}: &ember.Element{
					Path:        "1",
					ElementType: "node",
					Identifier:  "R3LAYVirtualPatchBay",
					IsOnline:    true,
				},
			},
			false,
		},
		{
			"+multiFirstCorrect",
			args{
				"1",
				3,
				[]readResponse{
					{
						data: []byte{
							0x60, 0x34, 0x6b, 0x32, 0xa0, 0x30, 0x63, 0x2e, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x27,
							0x31, 0x25, 0xa0, 0x16, 0x0c, 0x14, 0x52, 0x33, 0x4c, 0x41, 0x59, 0x56, 0x69, 0x72, 0x74,
							0x75, 0x61, 0x6c, 0x50, 0x61, 0x74, 0x63, 0x68, 0x42, 0x61, 0x79, 0xa1, 0x02, 0x0c, 0x00,
							0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xff,
						},
						err: nil,
					},
					//nolint:dupl
					{
						data: []byte{
							0x60, 0x82, 0x01, 0x01, 0x6b, 0x82, 0x00, 0xfd, 0xa0, 0x23, 0x69, 0x21, 0xa0, 0x05, 0x0d,
							0x03, 0x01, 0x02, 0x01, 0xa1, 0x18, 0x31, 0x16, 0xa0, 0x06, 0x0c, 0x04, 0x43, 0x61, 0x6c,
							0x6c, 0xad, 0x03, 0x02, 0x01, 0x03, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa2, 0x02, 0x0c, 0x00,
							0xa0, 0x82, 0x00, 0x81, 0x69, 0x7f, 0xa0, 0x05, 0x0d, 0x03, 0x01, 0x02, 0x02, 0xa1, 0x76,
							0x31, 0x74, 0xa0, 0x08, 0x0c, 0x06, 0x52, 0x65, 0x74, 0x75, 0x72, 0x6e, 0xad, 0x03, 0x02,
							0x01, 0x03, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa2, 0x5e, 0x0c, 0x5c, 0x5b, 0x7b, 0x22, 0x66,
							0x6e, 0x22, 0x3a, 0x22, 0x63, 0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 0x74, 0x6f, 0x67, 0x75,
							0x69, 0x22, 0x2c, 0x22, 0x66, 0x6e, 0x49, 0x44, 0x22, 0x3a, 0x22, 0x31, 0x31, 0x30, 0x38,
							0x22, 0x2c, 0x22, 0x66, 0x6e, 0x52, 0x65, 0x74, 0x54, 0x79, 0x70, 0x65, 0x22, 0x3a, 0x22,
							0x4a, 0x53, 0x4f, 0x4e, 0x22, 0x2c, 0x22, 0x66, 0x6e, 0x52, 0x65, 0x74, 0x56, 0x61, 0x6c,
							0x75, 0x65, 0x22, 0x3a, 0x5b, 0x22, 0x73, 0x74, 0x61, 0x74, 0x65, 0x22, 0x3a, 0x22, 0x63,
							0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 0x69, 0x6e, 0x67, 0x22, 0x5d, 0x7d, 0x5d, 0xa0, 0x51,
							0x74, 0x4f, 0xa0, 0x05, 0x0d, 0x03, 0x01, 0x02, 0x03, 0xa1, 0x46, 0x31, 0x44, 0xa0, 0x0e,
							0x0c, 0x0c, 0x43, 0x61, 0x6c, 0x6c, 0x46, 0x72, 0x6f, 0x6d, 0x4a, 0x53, 0x4f, 0x4e, 0xa2,
							0x17, 0x30, 0x15, 0xa0, 0x13, 0x75, 0x11, 0xa0, 0x03, 0x02, 0x01, 0x03, 0xa1, 0x0a, 0x0c,
							0x08, 0x70, 0x61, 0x72, 0x73, 0x4a, 0x53, 0x4f, 0x4e, 0xa3, 0x19, 0x30, 0x17, 0xa0, 0x15,
							0x75, 0x13, 0xa0, 0x03, 0x02, 0x01, 0x03, 0xa1, 0x0c, 0x0c, 0x0a, 0x72, 0x65, 0x73, 0x75,
							0x6c, 0x74, 0x4a, 0x53, 0x4f, 0x4e,
						},
						err: nil,
					},
				},
			},
			ember.ElementCollection{
				ember.ElementKey{
					ID: "R3LAYVirtualPatchBay", Path: "1",
				}: &ember.Element{
					Path:        "1",
					ElementType: "node",
					Identifier:  "R3LAYVirtualPatchBay",
					IsOnline:    true,
				},
			},
			false,
		},
		{
			"+multiSecondCorrect",
			args{
				"1",
				3,
				[]readResponse{
					//nolint:dupl
					{
						data: []byte{
							0x60, 0x82, 0x01, 0x01, 0x6b, 0x82, 0x00, 0xfd, 0xa0, 0x23, 0x69, 0x21, 0xa0, 0x05, 0x0d,
							0x03, 0x01, 0x02, 0x01, 0xa1, 0x18, 0x31, 0x16, 0xa0, 0x06, 0x0c, 0x04, 0x43, 0x61, 0x6c,
							0x6c, 0xad, 0x03, 0x02, 0x01, 0x03, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa2, 0x02, 0x0c, 0x00,
							0xa0, 0x82, 0x00, 0x81, 0x69, 0x7f, 0xa0, 0x05, 0x0d, 0x03, 0x01, 0x02, 0x02, 0xa1, 0x76,
							0x31, 0x74, 0xa0, 0x08, 0x0c, 0x06, 0x52, 0x65, 0x74, 0x75, 0x72, 0x6e, 0xad, 0x03, 0x02,
							0x01, 0x03, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa2, 0x5e, 0x0c, 0x5c, 0x5b, 0x7b, 0x22, 0x66,
							0x6e, 0x22, 0x3a, 0x22, 0x63, 0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 0x74, 0x6f, 0x67, 0x75,
							0x69, 0x22, 0x2c, 0x22, 0x66, 0x6e, 0x49, 0x44, 0x22, 0x3a, 0x22, 0x31, 0x31, 0x30, 0x38,
							0x22, 0x2c, 0x22, 0x66, 0x6e, 0x52, 0x65, 0x74, 0x54, 0x79, 0x70, 0x65, 0x22, 0x3a, 0x22,
							0x4a, 0x53, 0x4f, 0x4e, 0x22, 0x2c, 0x22, 0x66, 0x6e, 0x52, 0x65, 0x74, 0x56, 0x61, 0x6c,
							0x75, 0x65, 0x22, 0x3a, 0x5b, 0x22, 0x73, 0x74, 0x61, 0x74, 0x65, 0x22, 0x3a, 0x22, 0x63,
							0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 0x69, 0x6e, 0x67, 0x22, 0x5d, 0x7d, 0x5d, 0xa0, 0x51,
							0x74, 0x4f, 0xa0, 0x05, 0x0d, 0x03, 0x01, 0x02, 0x03, 0xa1, 0x46, 0x31, 0x44, 0xa0, 0x0e,
							0x0c, 0x0c, 0x43, 0x61, 0x6c, 0x6c, 0x46, 0x72, 0x6f, 0x6d, 0x4a, 0x53, 0x4f, 0x4e, 0xa2,
							0x17, 0x30, 0x15, 0xa0, 0x13, 0x75, 0x11, 0xa0, 0x03, 0x02, 0x01, 0x03, 0xa1, 0x0a, 0x0c,
							0x08, 0x70, 0x61, 0x72, 0x73, 0x4a, 0x53, 0x4f, 0x4e, 0xa3, 0x19, 0x30, 0x17, 0xa0, 0x15,
							0x75, 0x13, 0xa0, 0x03, 0x02, 0x01, 0x03, 0xa1, 0x0c, 0x0c, 0x0a, 0x72, 0x65, 0x73, 0x75,
							0x6c, 0x74, 0x4a, 0x53, 0x4f, 0x4e,
						},
						err: nil,
					},
					{
						data: []byte{
							0x60, 0x34, 0x6b, 0x32, 0xa0, 0x30, 0x63, 0x2e, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x27,
							0x31, 0x25, 0xa0, 0x16, 0x0c, 0x14, 0x52, 0x33, 0x4c, 0x41, 0x59, 0x56, 0x69, 0x72, 0x74,
							0x75, 0x61, 0x6c, 0x50, 0x61, 0x74, 0x63, 0x68, 0x42, 0x61, 0x79, 0xa1, 0x02, 0x0c, 0x00,
							0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xff,
						},
						err: nil,
					},
				},
			},
			ember.ElementCollection{
				ember.ElementKey{
					ID: "R3LAYVirtualPatchBay", Path: "1",
				}: &ember.Element{
					Path:        "1",
					ElementType: "node",
					Identifier:  "R3LAYVirtualPatchBay",
					IsOnline:    true,
				},
			},
			false,
		},
		{
			"+multiFirstCorrectWithInvalidData",
			args{
				"1",
				3,
				[]readResponse{
					{
						data: []byte{
							0x60, 0x34, 0x6b, 0x32, 0xa0, 0x30, 0x63, 0x2e, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x27,
							0x31, 0x25, 0xa0, 0x16, 0x0c, 0x14, 0x52, 0x33, 0x4c, 0x41, 0x59, 0x56, 0x69, 0x72, 0x74,
							0x75, 0x61, 0x6c, 0x50, 0x61, 0x74, 0x63, 0x68, 0x42, 0x61, 0x79, 0xa1, 0x02, 0x0c, 0x00,
							0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xff,
						},
						err: nil,
					},
					{
						data: []byte{
							0x00, 0x82, 0x01, 0x01, 0x6b, 0x82, 0x00, 0xfd, 0xa0, 0x23, 0x69, 0x21, 0xa0, 0x05, 0x0d,
							0x03, 0x01, 0x02, 0x01, 0xa1, 0x18, 0x31, 0x16, 0xa0, 0x06, 0x0c, 0x04, 0x43, 0x61, 0x6c,
							0x6c, 0xad, 0x03, 0x02, 0x01, 0x03, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa2, 0x02, 0x0c, 0x00,
							0xa0, 0x82, 0x00, 0x81, 0x69, 0x7f, 0xa0, 0x05, 0x0d, 0x03, 0x01, 0x02, 0x02, 0xa1, 0x76,
							0x31, 0x74, 0xa0, 0x08, 0x0c, 0x06, 0x52, 0x65, 0x74, 0x75, 0x72, 0x6e, 0xad, 0x03, 0x02,
							0x01, 0x03, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa2, 0x5e, 0x0c, 0x5c, 0x5b, 0x7b, 0x22, 0x66,
							0x6e, 0x22, 0x3a, 0x22, 0x63, 0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 0x74, 0x6f, 0x67, 0x75,
							0x69, 0x22, 0x2c,
						},
						err: nil,
					},
				},
			},
			ember.ElementCollection{
				ember.ElementKey{
					ID: "R3LAYVirtualPatchBay", Path: "1",
				}: &ember.Element{
					Path:        "1",
					ElementType: "node",
					Identifier:  "R3LAYVirtualPatchBay",
					IsOnline:    true,
				},
			},
			false,
		},
		{
			"+multiSecondCorrectWithInvalidData",
			args{
				"1",
				3,
				[]readResponse{
					{
						data: []byte{
							0x00, 0x82, 0x01, 0x01, 0x6b, 0x82, 0x00, 0xfd, 0xa0, 0x23, 0x69, 0x21, 0xa0, 0x05, 0x0d,
							0x03, 0x01, 0x02, 0x01, 0xa1, 0x18, 0x31, 0x16, 0xa0, 0x06, 0x0c, 0x04, 0x43, 0x61, 0x6c,
							0x6c, 0xad, 0x03, 0x02, 0x01, 0x03, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa2, 0x02, 0x0c, 0x00,
							0xa0, 0x82, 0x00, 0x81, 0x69, 0x7f, 0xa0, 0x05, 0x0d, 0x03, 0x01, 0x02, 0x02, 0xa1, 0x76,
							0x31, 0x74, 0xa0, 0x08, 0x0c, 0x06, 0x52, 0x65, 0x74, 0x75, 0x72, 0x6e, 0xad, 0x03, 0x02,
							0x01, 0x03, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa2, 0x5e, 0x0c, 0x5c, 0x5b, 0x7b, 0x22, 0x66,
							0x6e, 0x22, 0x3a, 0x22, 0x63, 0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 0x74, 0x6f, 0x67, 0x75,
							0x69, 0x22, 0x2c,
						},
						err: nil,
					},
					{
						data: []byte{
							0x60, 0x34, 0x6b, 0x32, 0xa0, 0x30, 0x63, 0x2e, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x27,
							0x31, 0x25, 0xa0, 0x16, 0x0c, 0x14, 0x52, 0x33, 0x4c, 0x41, 0x59, 0x56, 0x69, 0x72, 0x74,
							0x75, 0x61, 0x6c, 0x50, 0x61, 0x74, 0x63, 0x68, 0x42, 0x61, 0x79, 0xa1, 0x02, 0x0c, 0x00,
							0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xff,
						},
						err: nil,
					},
				},
			},
			ember.ElementCollection{
				ember.ElementKey{
					ID: "R3LAYVirtualPatchBay", Path: "1",
				}: &ember.Element{
					Path:        "1",
					ElementType: "node",
					Identifier:  "R3LAYVirtualPatchBay",
					IsOnline:    true,
				},
			},
			false,
		},
		{
			"-readErr",
			args{
				"1",
				3,
				[]readResponse{
					{
						err: errs.New("fail"),
					},
				},
			},
			nil,
			true,
		},
		{
			"-timeout",
			args{
				"1",
				1,
				nil,
			},
			nil,
			true,
		},
	}

	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			ch := &connHandler{
				logr:       log.New("Test"),
				readData:   make(chan readResponse),
				parsedData: make(chan parsedResponse),
			}

			go func() {
				for _, rr := range tt.args.message {
					ch.readData <- rr
				}
			}()

			got := ch.readExpected(tt.args.path, time.Duration(tt.args.timeout)*time.Second)
			if (got.err != nil) != tt.wantErr {
				t.Fatalf("connHandler.readExpected() error = %v, wantErr %v", got.err, tt.wantErr)
			}

			if diff := cmp.Diff(tt.want, got.element); diff != "" {
				t.Fatalf("connHandler.readExpected() = %s", diff)
			}
		})
	}
}
