/*
** Copyright (C) 2001-2025 Zabbix SIA
**
** This program is free software: you can redistribute it and/or modify it under the terms of
** the GNU Affero General Public License as published by the Free Software Foundation, version 3.
**
** This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
** without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
** See the GNU Affero General Public License for more details.
**
** You should have received a copy of the GNU Affero General Public License along with this program.
** If not, see <https://www.gnu.org/licenses/>.
**/

package s101

import (
	"testing"

	"github.com/google/go-cmp/cmp"
)

func TestEncode(t *testing.T) {
	t.Parallel()

	type args struct {
		message    []byte
		packetType uint8
	}

	tests := []struct {
		name string
		args args
		want []uint8
	}{
		{
			"+valid",
			args{
				[]byte{
					0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x62, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x20, 0xa1, 0x03, 0x02,
					0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				},
				0x60,
			},
			[]byte{
				0xfe, 0x00, 0x0e, 0x00, 0x01, 0x60, 0x01, 0x02, 0x28, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x62,
				0x80, 0xa0, 0x03, 0x02, 0x01, 0x20, 0xa1, 0x03, 0x02, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x09, 0x0d, 0xff,
			},
		},
		{
			"+multiPacket",
			args{
				[]byte{
					0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x62, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x20, 0xa1, 0x03, 0x02,
					0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				},
				0x80,
			},
			[]byte{
				0xfe, 0x00, 0x0e, 0x00, 0x01, 0x80, 0x01, 0x02, 0x28, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x62,
				0x80, 0xa0, 0x03, 0x02, 0x01, 0x20, 0xa1, 0x03, 0x02, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x50, 0x30, 0xff, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0x40, 0x01, 0x02, 0x28, 0x02, 0x01,
				0x23, 0xff,
			},
		},
	}

	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got := Encode(tt.args.message, tt.args.packetType)
			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("Encode() = %s", diff)
			}
		})
	}
}

func TestDecode(t *testing.T) {
	t.Parallel()

	type args struct {
		message [][]byte
	}

	tests := []struct {
		name     string
		args     args
		wantData []byte
		wantType byte
		wantErr  bool
	}{
		{
			"+valid",
			args{
				[][]byte{
					{
						0xfe, 0x00, 0x0e, 0x00, 0x01, 0xc0, 0x01, 0x02, 0x1e, 0x02, 0x60, 0x34, 0x6b, 0x32, 0xa0, 0x30,
						0x63, 0x2e, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x27, 0x31, 0x25, 0xa0, 0x16, 0x0c, 0x14, 0x52,
						0x33, 0x4c, 0x41, 0x59, 0x56, 0x69, 0x72, 0x74, 0x75, 0x61, 0x6c, 0x50, 0x61, 0x74, 0x63, 0x68,
						0x42, 0x61, 0x79, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xfd,
						0xdf, 0xb5, 0x11, 0xff,
					},
				},
			},
			[]byte{
				0x60, 0x34, 0x6b, 0x32, 0xa0, 0x30, 0x63, 0x2e, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x27, 0x31, 0x25,
				0xa0, 0x16, 0x0c, 0x14, 0x52, 0x33, 0x4c, 0x41, 0x59, 0x56, 0x69, 0x72, 0x74, 0x75, 0x61, 0x6c, 0x50,
				0x61, 0x74, 0x63, 0x68, 0x42, 0x61, 0x79, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03,
				0x01, 0x01, 0xff,
			},
			SinglePacket,
			false,
		},
		{
			"+functionRequest",
			args{
				[][]byte{
					{
						0xfe, 0x00, 0x0e, 0x00, 0x01, 0xc0, 0x01, 0x02, 0x1e, 0x02, 0x60, 0x82, 0x01, 0x01, 0x6b, 0x82,
						0x00, 0xfd, 0xdd, 0xa0, 0x23, 0x69, 0x21, 0xa0, 0x05, 0x0d, 0x03, 0x01, 0x02, 0x01, 0xa1, 0x18,
						0x31, 0x16, 0xa0, 0x06, 0x0c, 0x04, 0x43, 0x61, 0x6c, 0x6c, 0xad, 0x03, 0x02, 0x01, 0x03, 0xa5,
						0x03, 0x02, 0x01, 0x03, 0xa2, 0x02, 0x0c, 0x00, 0xa0, 0x82, 0x00, 0x81, 0x69, 0x7f, 0xa0, 0x05,
						0x0d, 0x03, 0x01, 0x02, 0x02, 0xa1, 0x76, 0x31, 0x74, 0xa0, 0x08, 0x0c, 0x06, 0x52, 0x65, 0x74,
						0x75, 0x72, 0x6e, 0xad, 0x03, 0x02, 0x01, 0x03, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa2, 0x5e, 0x0c,
						0x5c, 0x5b, 0x7b, 0x22, 0x66, 0x6e, 0x22, 0x3a, 0x22, 0x63, 0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74,
						0x74, 0x6f, 0x67, 0x75, 0x69, 0x22, 0x2c, 0x22, 0x66, 0x6e, 0x49, 0x44, 0x22, 0x3a, 0x22, 0x31,
						0x31, 0x30, 0x38, 0x22, 0x2c, 0x22, 0x66, 0x6e, 0x52, 0x65, 0x74, 0x54, 0x79, 0x70, 0x65, 0x22,
						0x3a, 0x22, 0x4a, 0x53, 0x4f, 0x4e, 0x22, 0x2c, 0x22, 0x66, 0x6e, 0x52, 0x65, 0x74, 0x56, 0x61,
						0x6c, 0x75, 0x65, 0x22, 0x3a, 0x5b, 0x22, 0x73, 0x74, 0x61, 0x74, 0x65, 0x22, 0x3a, 0x22, 0x63,
						0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 0x69, 0x6e, 0x67, 0x22, 0x5d, 0x7d, 0x5d, 0xa0, 0x51, 0x74,
						0x4f, 0xa0, 0x05, 0x0d, 0x03, 0x01, 0x02, 0x03, 0xa1, 0x46, 0x31, 0x44, 0xa0, 0x0e, 0x0c, 0x0c,
						0x43, 0x61, 0x6c, 0x6c, 0x46, 0x72, 0x6f, 0x6d, 0x4a, 0x53, 0x4f, 0x4e, 0xa2, 0x17, 0x30, 0x15,
						0xa0, 0x13, 0x75, 0x11, 0xa0, 0x03, 0x02, 0x01, 0x03, 0xa1, 0x0a, 0x0c, 0x08, 0x70, 0x61, 0x72,
						0x73, 0x4a, 0x53, 0x4f, 0x4e, 0xa3, 0x19, 0x30, 0x17, 0xa0, 0x15, 0x75, 0x13, 0xa0, 0x03, 0x02,
						0x01, 0x03, 0xa1, 0x0c, 0x0c, 0x0a, 0x72, 0x65, 0x73, 0x75, 0x6c, 0x74, 0x4a, 0x53, 0x4f, 0x4e,
						0xfd, 0xdc, 0x8a, 0xff,
					},
				},
			},
			[]byte{
				0x60, 0x82, 0x01, 0x01, 0x6b, 0x82, 0x00, 0xfd, 0xa0, 0x23, 0x69, 0x21, 0xa0, 0x05, 0x0d, 0x03,
				0x01, 0x02, 0x01, 0xa1, 0x18, 0x31, 0x16, 0xa0, 0x06, 0x0c, 0x04, 0x43, 0x61, 0x6c, 0x6c, 0xad,
				0x03, 0x02, 0x01, 0x03, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa2, 0x02, 0x0c, 0x00, 0xa0, 0x82, 0x00,
				0x81, 0x69, 0x7f, 0xa0, 0x05, 0x0d, 0x03, 0x01, 0x02, 0x02, 0xa1, 0x76, 0x31, 0x74, 0xa0, 0x08,
				0x0c, 0x06, 0x52, 0x65, 0x74, 0x75, 0x72, 0x6e, 0xad, 0x03, 0x02, 0x01, 0x03, 0xa5, 0x03, 0x02,
				0x01, 0x03, 0xa2, 0x5e, 0x0c, 0x5c, 0x5b, 0x7b, 0x22, 0x66, 0x6e, 0x22, 0x3a, 0x22, 0x63, 0x6f,
				0x6e, 0x6e, 0x65, 0x63, 0x74, 0x74, 0x6f, 0x67, 0x75, 0x69, 0x22, 0x2c, 0x22, 0x66, 0x6e, 0x49,
				0x44, 0x22, 0x3a, 0x22, 0x31, 0x31, 0x30, 0x38, 0x22, 0x2c, 0x22, 0x66, 0x6e, 0x52, 0x65, 0x74,
				0x54, 0x79, 0x70, 0x65, 0x22, 0x3a, 0x22, 0x4a, 0x53, 0x4f, 0x4e, 0x22, 0x2c, 0x22, 0x66, 0x6e,
				0x52, 0x65, 0x74, 0x56, 0x61, 0x6c, 0x75, 0x65, 0x22, 0x3a, 0x5b, 0x22, 0x73, 0x74, 0x61, 0x74,
				0x65, 0x22, 0x3a, 0x22, 0x63, 0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 0x69, 0x6e, 0x67, 0x22, 0x5d,
				0x7d, 0x5d, 0xa0, 0x51, 0x74, 0x4f, 0xa0, 0x05, 0x0d, 0x03, 0x01, 0x02, 0x03, 0xa1, 0x46, 0x31,
				0x44, 0xa0, 0x0e, 0x0c, 0x0c, 0x43, 0x61, 0x6c, 0x6c, 0x46, 0x72, 0x6f, 0x6d, 0x4a, 0x53, 0x4f,
				0x4e, 0xa2, 0x17, 0x30, 0x15, 0xa0, 0x13, 0x75, 0x11, 0xa0, 0x03, 0x02, 0x01, 0x03, 0xa1, 0x0a,
				0x0c, 0x08, 0x70, 0x61, 0x72, 0x73, 0x4a, 0x53, 0x4f, 0x4e, 0xa3, 0x19, 0x30, 0x17, 0xa0, 0x15,
				0x75, 0x13, 0xa0, 0x03, 0x02, 0x01, 0x03, 0xa1, 0x0c, 0x0c, 0x0a, 0x72, 0x65, 0x73, 0x75, 0x6c,
				0x74, 0x4a, 0x53, 0x4f, 0x4e,
			},
			SinglePacket,
			false,
		},
		{
			"+bigCollection",
			args{
				[][]byte{
					{
						0xfe, 0x00, 0x0e, 0x00,
						0x01, 0xc0, 0x01, 0x02, 0x1e, 0x02, 0x60, 0x82, 0x01, 0xfd, 0xdd, 0x6b, 0x82, 0x01, 0xfd, 0xd9,
						0xa0, 0x40, 0x6a, 0x3e, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x00, 0xa1, 0x36, 0x31, 0x34, 0xa0, 0x0a,
						0x0c, 0x08, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x1d,
						0x0c, 0x1b, 0x64, 0x65, 0x2e, 0x6c, 0x2d, 0x73, 0x2d, 0x62, 0x2e, 0x65, 0x6d, 0x62, 0x65, 0x72,
						0x70, 0x6c, 0x75, 0x73, 0x2e, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0xa3, 0x03, 0x01,
						0x01, 0xfd, 0xdf, 0xa0, 0x29, 0x6a, 0x27, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x01, 0xa1, 0x1f, 0x31,
						0x1d, 0xa0, 0x0e, 0x0c, 0x0c, 0x45, 0x6e, 0x76, 0x69, 0x72, 0x6f, 0x6e, 0x6d, 0x65, 0x6e, 0x74,
						0x73, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xa0,
						0x25, 0x6a, 0x23, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x02, 0xa1, 0x1b, 0x31, 0x19, 0xa0, 0x0a, 0x0c,
						0x08, 0x46, 0x75, 0x6e, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c,
						0x00, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xa0, 0x24, 0x6a, 0x22, 0xa0, 0x04, 0x0d, 0x02, 0x01,
						0x03, 0xa1, 0x1a, 0x31, 0x18, 0xa0, 0x09, 0x0c, 0x07, 0x53, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x73,
						0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xa0, 0x22,
						0x6a, 0x20, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x04, 0xa1, 0x18, 0x31, 0x16, 0xa0, 0x07, 0x0c, 0x05,
						0x53, 0x69, 0x6e, 0x6b, 0x73, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01,
						0x01, 0xfd, 0xdf, 0xa0, 0x28, 0x6a, 0x26, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x05, 0xa1, 0x1e, 0x31,
						0x1c, 0xa0, 0x0d, 0x0c, 0x0b, 0x43, 0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x73,
						0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xa0, 0x2d,
						0x6a, 0x2b, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x06, 0xa1, 0x23, 0x31, 0x21, 0xa0, 0x12, 0x0c, 0x10,
						0x53, 0x69, 0x6e, 0x6b, 0x53, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x4d, 0x61, 0x74, 0x72, 0x69, 0x78,
						0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xa0, 0x29,
						0x6a, 0x27, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x07, 0xa1, 0x1f, 0x31, 0x1d, 0xa0, 0x0e, 0x0c, 0x0c,
						0x41, 0x75, 0x64, 0x69, 0x6f, 0x46, 0x69, 0x6c, 0x74, 0x65, 0x72, 0x73, 0xa1, 0x02, 0x0c, 0x00,
						0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xa0, 0x26, 0x6a, 0x24, 0xa0, 0x04,
						0x0d, 0x02, 0x01, 0x08, 0xa1, 0x1c, 0x31, 0x1a, 0xa0, 0x0b, 0x0c, 0x09, 0x53, 0x74, 0x72, 0x65,
						0x61, 0x6d, 0x69, 0x6e, 0x67, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01,
						0x01, 0xfd, 0xdf, 0xa0, 0x24, 0x6a, 0x22, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x09, 0xa1, 0x1a, 0x31,
						0x18, 0xa0, 0x09, 0x0c, 0x07, 0x4c, 0x6f, 0x67, 0x69, 0x63, 0x61, 0x6c, 0xa1, 0x02, 0x0c, 0x00,
						0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xa0, 0x21, 0x6a, 0x1f, 0xa0, 0x04,
						0x0d, 0x02, 0x01, 0x0b, 0xa1, 0x17, 0x31, 0x15, 0xa0, 0x06, 0x0c, 0x04, 0x47, 0x50, 0x49, 0x4f,
						0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xfd, 0xdf, 0xa0, 0x24,
						0x6a, 0x22, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x0f, 0xa1, 0x1a, 0x31, 0x18, 0xa0, 0x09, 0x0c, 0x07,
						0x4c, 0x69, 0x63, 0x65, 0x6e, 0x73, 0x65, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3,
						0x03, 0x01, 0x01, 0xfd, 0xdf, 0xb4, 0x70, 0xff,
					},
				},
			},
			[]byte{
				0x60, 0x82, 0x01, 0xfd, 0x6b, 0x82, 0x01, 0xf9, 0xa0, 0x40, 0x6a, 0x3e, 0xa0, 0x04, 0x0d, 0x02,
				0x01, 0x00, 0xa1, 0x36, 0x31, 0x34, 0xa0, 0x0a, 0x0c, 0x08, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69,
				0x74, 0x79, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x1d, 0x0c, 0x1b, 0x64, 0x65, 0x2e, 0x6c, 0x2d, 0x73,
				0x2d, 0x62, 0x2e, 0x65, 0x6d, 0x62, 0x65, 0x72, 0x70, 0x6c, 0x75, 0x73, 0x2e, 0x69, 0x64, 0x65,
				0x6e, 0x74, 0x69, 0x74, 0x79, 0xa3, 0x03, 0x01, 0x01, 0xff, 0xa0, 0x29, 0x6a, 0x27, 0xa0, 0x04,
				0x0d, 0x02, 0x01, 0x01, 0xa1, 0x1f, 0x31, 0x1d, 0xa0, 0x0e, 0x0c, 0x0c, 0x45, 0x6e, 0x76, 0x69,
				0x72, 0x6f, 0x6e, 0x6d, 0x65, 0x6e, 0x74, 0x73, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00,
				0xa3, 0x03, 0x01, 0x01, 0xff, 0xa0, 0x25, 0x6a, 0x23, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x02, 0xa1,
				0x1b, 0x31, 0x19, 0xa0, 0x0a, 0x0c, 0x08, 0x46, 0x75, 0x6e, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0xa1,
				0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xff, 0xa0, 0x24, 0x6a, 0x22,
				0xa0, 0x04, 0x0d, 0x02, 0x01, 0x03, 0xa1, 0x1a, 0x31, 0x18, 0xa0, 0x09, 0x0c, 0x07, 0x53, 0x6f,
				0x75, 0x72, 0x63, 0x65, 0x73, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01,
				0x01, 0xff, 0xa0, 0x22, 0x6a, 0x20, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x04, 0xa1, 0x18, 0x31, 0x16,
				0xa0, 0x07, 0x0c, 0x05, 0x53, 0x69, 0x6e, 0x6b, 0x73, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c,
				0x00, 0xa3, 0x03, 0x01, 0x01, 0xff, 0xa0, 0x28, 0x6a, 0x26, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x05,
				0xa1, 0x1e, 0x31, 0x1c, 0xa0, 0x0d, 0x0c, 0x0b, 0x43, 0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 0x69,
				0x6f, 0x6e, 0x73, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xff,
				0xa0, 0x2d, 0x6a, 0x2b, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x06, 0xa1, 0x23, 0x31, 0x21, 0xa0, 0x12,
				0x0c, 0x10, 0x53, 0x69, 0x6e, 0x6b, 0x53, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x4d, 0x61, 0x74, 0x72,
				0x69, 0x78, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xff, 0xa0,
				0x29, 0x6a, 0x27, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x07, 0xa1, 0x1f, 0x31, 0x1d, 0xa0, 0x0e, 0x0c,
				0x0c, 0x41, 0x75, 0x64, 0x69, 0x6f, 0x46, 0x69, 0x6c, 0x74, 0x65, 0x72, 0x73, 0xa1, 0x02, 0x0c,
				0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xff, 0xa0, 0x26, 0x6a, 0x24, 0xa0, 0x04,
				0x0d, 0x02, 0x01, 0x08, 0xa1, 0x1c, 0x31, 0x1a, 0xa0, 0x0b, 0x0c, 0x09, 0x53, 0x74, 0x72, 0x65,
				0x61, 0x6d, 0x69, 0x6e, 0x67, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01,
				0x01, 0xff, 0xa0, 0x24, 0x6a, 0x22, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x09, 0xa1, 0x1a, 0x31, 0x18,
				0xa0, 0x09, 0x0c, 0x07, 0x4c, 0x6f, 0x67, 0x69, 0x63, 0x61, 0x6c, 0xa1, 0x02, 0x0c, 0x00, 0xa4,
				0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xff, 0xa0, 0x21, 0x6a, 0x1f, 0xa0, 0x04, 0x0d, 0x02,
				0x01, 0x0b, 0xa1, 0x17, 0x31, 0x15, 0xa0, 0x06, 0x0c, 0x04, 0x47, 0x50, 0x49, 0x4f, 0xa1, 0x02,
				0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xff, 0xa0, 0x24, 0x6a, 0x22, 0xa0,
				0x04, 0x0d, 0x02, 0x01, 0x0f, 0xa1, 0x1a, 0x31, 0x18, 0xa0, 0x09, 0x0c, 0x07, 0x4c, 0x69, 0x63,
				0x65, 0x6e, 0x73, 0x65, 0xa1, 0x02, 0x0c, 0x00, 0xa4, 0x02, 0x0c, 0x00, 0xa3, 0x03, 0x01, 0x01,
				0xff,
			},
			SinglePacket,
			false,
		},
		{
			"+noLengthByte",
			args{
				[][]byte{
					{
						0xfe, 0x00, 0x0e, 0x00, 0x01, 0xc0, 0x01, 0x02, 0x14, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80,
						0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x1d, 0x06, 0x02, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa2,
						0x0a, 0x0c, 0x08, 0x34, 0x30, 0x20, 0x64, 0x65, 0x67, 0x20, 0x43, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4b, 0x5e, 0xff,
					},
				},
			},
			[]byte{
				0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x69, 0x80, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x1d, 0x06, 0x02,
				0x01, 0xa1, 0x80, 0x31, 0x80, 0xa2, 0x0a, 0x0c, 0x08, 0x34, 0x30, 0x20, 0x64, 0x65, 0x67, 0x20,
				0x43, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
			SinglePacket,
			false,
		},
		{
			"+multipleGlow",
			args{
				[][]byte{
					{
						0xfe, 0x00, 0x0e, 0x00, 0x01, 0x80, 0x01, 0x02, 0x1e, 0x02, 0x60, 0x82, 0x04, 0x23, 0x6b, 0x82,
						0x04, 0x1f, 0xa0, 0x25, 0x6a, 0x23, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x08, 0x03, 0x01, 0x07, 0xa1,
						0x18, 0x31, 0x16, 0xa0, 0x07, 0x0c, 0x05, 0x53, 0x74, 0x61, 0x03, 0x02, 0x01, 0x03, 0xa1, 0x0a,
						0x0c, 0x08, 0x70, 0x61, 0x72, 0x73, 0x4a, 0x53, 0x4f, 0x4e, 0xa3, 0x19, 0x30, 0x17, 0xa0, 0x15,
						0x75, 0x13, 0xa0, 0x03, 0x02, 0x01, 0x03, 0xa1, 0x0c, 0x0c, 0x0a, 0x72, 0x65, 0x73, 0x75, 0x6c,
						0x74, 0x4a, 0x53, 0x4f, 0x4e, 0xa0, 0x3c, 0x74, 0x3a, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x08, 0x03,
						0x01, 0x05, 0xa1, 0x2f, 0x31, 0x2d, 0xa0, 0x10, 0x0c, 0x0e, 0x5c, 0x5b, 0xff,
					},
					{
						0xfe, 0x00, 0x0e, 0x00, 0x01, 0x40, 0x01, 0x02, 0x1e, 0x02, 0x64, 0x65, 0x6c, 0x65, 0x74, 0x65,
						0x46, 0x72, 0x6f, 0x6d, 0x4a, 0x53, 0x4f, 0x4e, 0xa3, 0x19, 0x30, 0x17, 0xa0, 0x15, 0x75, 0x13,
						0xa0, 0x03, 0x02, 0x01, 0x03, 0xa1, 0x0c, 0x0c, 0x0a, 0x72, 0x65, 0x73, 0x75, 0x6c, 0x74, 0x4a,
						0x53, 0x4f, 0x4e, 0x01, 0x4f, 0xff,
					},
				},
			},
			[]byte{
				0x60, 0x82, 0x04, 0x23, 0x6b, 0x82, 0x04, 0x1f, 0xa0, 0x25, 0x6a, 0x23, 0xa0, 0x07, 0x0d, 0x05, 0x01,
				0x08, 0x03, 0x01, 0x07, 0xa1, 0x18, 0x31, 0x16, 0xa0, 0x07, 0x0c, 0x05, 0x53, 0x74, 0x61, 0x03, 0x02,
				0x01, 0x03, 0xa1, 0x0a, 0x0c, 0x08, 0x70, 0x61, 0x72, 0x73, 0x4a, 0x53, 0x4f, 0x4e, 0xa3, 0x19, 0x30,
				0x17, 0xa0, 0x15, 0x75, 0x13, 0xa0, 0x03, 0x02, 0x01, 0x03, 0xa1, 0x0c, 0x0c, 0x0a, 0x72, 0x65, 0x73,
				0x75, 0x6c, 0x74, 0x4a, 0x53, 0x4f, 0x4e, 0xa0, 0x3c, 0x74, 0x3a, 0xa0, 0x07, 0x0d, 0x05, 0x01, 0x08,
				0x03, 0x01, 0x05, 0xa1, 0x2f, 0x31, 0x2d, 0xa0, 0x10, 0x0c, 0x0e, 0x64, 0x65, 0x6c, 0x65, 0x74, 0x65,
				0x46, 0x72, 0x6f, 0x6d, 0x4a, 0x53, 0x4f, 0x4e, 0xa3, 0x19, 0x30, 0x17, 0xa0, 0x15, 0x75, 0x13, 0xa0,
				0x03, 0x02, 0x01, 0x03, 0xa1, 0x0c, 0x0c, 0x0a, 0x72, 0x65, 0x73, 0x75, 0x6c, 0x74, 0x4a, 0x53, 0x4f,
				0x4e,
			},
			LastMultiPacket,
			false,
		},
		{
			"-malformedGlow",
			args{[][]byte{{0xfe, 0x00, 0x0e, 0x00, 0x01, 0xc0, 0x01, 0x02, 0x1e, 0x02, 0x60, 0xff}}},
			nil,
			0,
			true,
		},
		{
			"-empty",
			args{[][]byte{}},
			nil,
			0,
			false,
		},
	}

	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got, got1, err := Decode(tt.args.message)
			if (err != nil) != tt.wantErr {
				t.Fatalf("Decode() error = %v, wantErr %v", err, tt.wantErr)
			}

			if diff := cmp.Diff(tt.wantData, got); diff != "" {
				t.Fatalf("Decode() = data %s", diff)
			}
			if diff := cmp.Diff(tt.wantType, got1); diff != "" {
				t.Fatalf("Decode() = type %s", diff)
			}
		})
	}
}

func Test_getS101s(t *testing.T) {
	t.Parallel()

	type args struct {
		in []uint8
	}

	tests := []struct {
		name           string
		args           args
		want           [][]uint8
		wantIncomplete []uint8
	}{
		{
			"+valid",
			args{[]byte{0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff}},
			[][]byte{{0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff}},
			nil,
		},
		{
			"+multiple",
			args{[]byte{0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xfe, 0x00, 0xff}},
			[][]byte{
				{0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff},
				{0xfe, 0x00, 0xff},
			},
			nil,
		},
		{
			"+multipleFESingleFF",
			args{[]byte{0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0xff}},
			[][]byte{
				{0xfe, 0x00, 0xff},
			},
			nil,
		},
		{
			"+onlyFE",
			args{[]byte{0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00}},
			nil,
			[]byte{0xfe, 0x00, 0x00, 0x00, 0x00, 0x00},
		},
		{
			"-onlyFF",
			args{[]byte{0x00, 0x00, 0xff, 0x00, 0x00}},
			nil,
			nil,
		},
		{
			"-noFForFE",
			args{[]byte{0x00, 0x00, 0xef, 0x00, 0x00}},
			nil,
			nil,
		},
		{
			"-empty",
			args{[]byte{}},
			nil,
			nil,
		},
	}

	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			full, incomplete := getS101s(tt.args.in)
			if diff := cmp.Diff(tt.want, full); diff != "" {
				t.Fatalf("getS101s() full = %s", full)
			}

			if diff := cmp.Diff(tt.wantIncomplete, incomplete); diff != "" {
				t.Fatalf("getS101s() incomplete = %s", diff)
			}
		})
	}
}

func Test_createS101(t *testing.T) {
	t.Parallel()

	type args struct {
		payload []byte
		pType   uint8
	}

	tests := []struct {
		name string
		args args
		want []byte
	}{
		{
			"+valid",
			args{
				[]byte{
					0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x62, 0x80,
					0xa0, 0x03, 0x02, 0x01, 0x20, 0xa1, 0x03, 0x02,
					0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00,
				},
				FirstMultiPacket,
			},
			[]byte{
				0xfe, 0x00, 0x0e, 0x00, 0x01, 0x80, 0x01, 0x02,
				0x28, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80,
				0x62, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x20, 0xa1,
				0x03, 0x02, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x30, 0xff,
			},
		},
		{
			"+emptyPayload",
			args{
				[]byte{},
				LastMultiPacket,
			},
			[]byte{
				0xfe, 0x00, 0x0e, 0x00, 0x01, 0x40, 0x01, 0x02,
				0x28, 0x02, 0x01, 0x23, 0xff,
			},
		},
	}

	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got := createS101(tt.args.payload, tt.args.pType)
			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("createS101() = %s", diff)
			}
		})
	}
}

func Test_escapeBytesAboveBOFNE(t *testing.T) {
	t.Parallel()

	type args struct {
		message []byte
	}

	tests := []struct {
		name string
		args args
		want []byte
	}{
		{
			"+valid",
			args{
				[]byte{
					0x00, 0x0e, 0x00, 1, 0x80, 1, 2, 40, 2, 0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x62, 0x80, 0xa0, 0x03,
					0x02, 0x01, 0x20, 0xa1, 0x03, 0x02, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				},
			},
			[]byte{
				0x00, 0x0e, 0x00, 1, 0x80, 1, 2, 40, 2, 0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x62, 0x80, 0xa0, 0x03,
				0x02, 0x01, 0x20, 0xa1, 0x03, 0x02, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
		},
		{
			"+escapeSingle",
			args{
				[]byte{
					0x02, 0x01, 0x20, 0xa1, 0x03, 0x02, 0x01, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				},
			},
			[]byte{
				0x02, 0x01, 0x20, 0xa1, 0x03, 0x02, 0x01, 0x00, 0x00, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			},
		},
		{
			"+escapeMultiple",
			args{
				[]byte{
					0x02, 0x01, 0xff, 0x00, 0x00, 0x02, 0x01, 0xff, 0x00, 0x00, 0x02, 0x01, 0xff, 0x00, 0x00,
				},
			},
			[]byte{
				0x02, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x02, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x02, 0x01, 0xfd, 0xdf, 0x00,
				0x00,
			},
		},
	}

	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got := escapeBytesAboveBOFNE(tt.args.message)
			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("escapeBytesAboveBOFNE() = %s", diff)
			}
		})
	}
}

func Test_getCRC(t *testing.T) {
	t.Parallel()

	type args struct {
		data []byte
	}

	tests := []struct {
		name string
		args args
		want []uint8
	}{
		{
			"+valid",
			args{
				[]byte{
					0x00, 0x0e, 0x00, 1, 0x80, 1, 2, 40, 2, 0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x62, 0x80, 0xa0, 0x03,
					0x02, 0x01, 0x20, 0xa1, 0x03, 0x02, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				},
			},
			[]byte{
				0x50, 0x30,
			},
		},
		{
			"+longByte",
			args{
				[]byte{
					0x00, 0x0e, 0x00, 0x01, 0xc0, 0x01, 0x02, 0x1e, 0x02, 0x60, 0x82, 0x01, 0x01, 0x6b, 0x82, 0x00,
					0xfd, 0xdd, 0xa0, 0x23, 0x69, 0x21, 0xa0, 0x05, 0x0d, 0x03, 0x01, 0x02, 0x01, 0xa1, 0x18, 0x31,
					0x16, 0xa0, 0x06, 0x0c, 0x04, 0x43, 0x61, 0x6c, 0x6c, 0xad, 0x03, 0x02, 0x01, 0x03, 0xa5, 0x03,
					0x02, 0x01, 0x03, 0xa2, 0x02, 0x0c, 0x00, 0xa0, 0x82, 0x00, 0x81, 0x69, 0x7f, 0xa0, 0x05, 0x0d,
					0x03, 0x01, 0x02, 0x02, 0xa1, 0x76, 0x31, 0x74, 0xa0, 0x08, 0x0c, 0x06, 0x52, 0x65, 0x74, 0x75,
					0x72, 0x6e, 0xad, 0x03, 0x02, 0x01, 0x03, 0xa5, 0x03, 0x02, 0x01, 0x03, 0xa2, 0x5e, 0x0c, 0x5c,
					0x5b, 0x7b, 0x22, 0x66, 0x6e, 0x22, 0x3a, 0x22, 0x63, 0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 0x74,
					0x6f, 0x67, 0x75, 0x69, 0x22, 0x2c, 0x22, 0x66, 0x6e, 0x49, 0x44, 0x22, 0x3a, 0x22, 0x31, 0x31,
					0x30, 0x38, 0x22, 0x2c, 0x22, 0x66, 0x6e, 0x52, 0x65, 0x74, 0x54, 0x79, 0x70, 0x65, 0x22, 0x3a,
					0x22, 0x4a, 0x53, 0x4f, 0x4e, 0x22, 0x2c, 0x22, 0x66, 0x6e, 0x52, 0x65, 0x74, 0x56, 0x61, 0x6c,
					0x75, 0x65, 0x22, 0x3a, 0x5b, 0x22, 0x73, 0x74, 0x61, 0x74, 0x65, 0x22, 0x3a, 0x22, 0x63, 0x6f,
					0x6e, 0x6e, 0x65, 0x63, 0x74, 0x69, 0x6e, 0x67, 0x22, 0x5d, 0x7d, 0x5d, 0xa0, 0x51, 0x74, 0x4f,
					0xa0, 0x05, 0x0d, 0x03, 0x01, 0x02, 0x03, 0xa1, 0x46, 0x31, 0x44, 0xa0, 0x0e, 0x0c, 0x0c, 0x43,
					0x61, 0x6c, 0x6c, 0x46, 0x72, 0x6f, 0x6d, 0x4a, 0x53, 0x4f, 0x4e, 0xa2, 0x17, 0x30, 0x15, 0xa0,
					0x13, 0x75, 0x11, 0xa0, 0x03, 0x02, 0x01, 0x03, 0xa1, 0x0a, 0x0c, 0x08, 0x70, 0x61, 0x72, 0x73,
					0x4a, 0x53, 0x4f, 0x4e, 0xa3, 0x19, 0x30, 0x17, 0xa0, 0x15, 0x75, 0x13, 0xa0, 0x03, 0x02, 0x01,
					0x03, 0xa1, 0x0c, 0x0c, 0x0a, 0x72, 0x65, 0x73, 0x75, 0x6c, 0x74, 0x4a, 0x53, 0x4f, 0x4e,
				},
			},
			[]byte{
				0xfd, 0xdc, 0x8a,
			},
		},
		{
			"+emptyPayload",
			args{
				[]byte{
					0x00, 0x0e, 0x00, 1, 0x60, 1, 2, 40, 2,
				},
			},
			[]byte{
				0x90, 0x43,
			},
		},
		{
			"+ceLastByte",
			args{
				[]byte{
					0x00, 0x0e, 0x00, 0xfd,
				},
			},
			[]byte{
				0xdc, 0x5c,
			},
		},
	}

	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got := getCRC(tt.args.data)
			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("getCRC() = %s", diff)
			}
		})
	}
}

func Test_parseCRC(t *testing.T) {
	t.Parallel()

	type args struct {
		in []uint8
	}

	tests := []struct {
		name string
		args args
		want []uint8
	}{
		{
			"+valid",
			args{[]byte{0x00, 0x01, 0x02, 0x03, 0x04}},
			[]byte{0x00, 0x01, 0x02, 0x03, 0x04},
		},
		{
			"+XOR",
			args{[]byte{0x00, 0x01, 0xff, 0x03, 0x04}},
			[]byte{0x00, 0x01, 0xfd, 0xdf, 0x03, 0x04},
		},
		{
			"+multipleXOR",
			args{[]byte{0x00, 0x01, 0xff, 0x03, 0xff, 0x04}},
			[]byte{0x00, 0x01, 0xfd, 0xdf, 0x03, 0xfd, 0xdf, 0x04},
		},
	}

	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got := parseCRC(tt.args.in)
			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("parseCRC() = %s", diff)
			}
		})
	}
}

func Test_computeCRCByte(t *testing.T) {
	t.Parallel()

	type args struct {
		crc uint16
		b   uint8
	}

	tests := []struct {
		name string
		args args
		want uint16
	}{
		{
			"+valid",
			args{0x01400, 0x04},
			17968,
		},
		{
			"+zeros",
			args{0, 0},
			0,
		},
	}

	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got := computeCRCByte(tt.args.crc, tt.args.b)
			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("computeCRCByte() = %s", diff)
			}
		})
	}
}
