/*
** Copyright (C) 2001-2024 Zabbix SIA
**
** This program is free software: you can redistribute it and/or modify it under the terms of
** the GNU Affero General Public License as published by the Free Software Foundation, version 3.
**
** This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
** without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
** See the GNU Affero General Public License for more details.
**
** You should have received a copy of the GNU Affero General Public License along with this program.
** If not, see <https://www.gnu.org/licenses/>.
**/

package ember

import (
	"testing"

	"github.com/google/go-cmp/cmp"
	"golang.zabbix.com/plugin/ember-plus/ember/asn1"
)

func TestElementCollection_Populate(t *testing.T) {
	t.Parallel()

	type args struct {
		data *asn1.Decoder
	}

	tests := []struct {
		name    string
		ec      ElementCollection
		args    args
		want    ElementCollection
		wantErr bool
	}{
		{
			"+valid",
			ElementCollection{},
			args{
				asn1.NewDecoder(
					[]byte{
						0x60, 0x34, 0x6B, 0x32, 0xA0, 0x30, 0x63, 0x2E, 0xA0, 0x03, 0x02, 0x01, 0x01, 0xA1, 0x27, 0x31,
						0x25, 0xA0, 0x16, 0x0C, 0x14, 0x52, 0x33, 0x4C, 0x41, 0x59, 0x56, 0x69, 0x72, 0x74, 0x75, 0x61,
						0x6C, 0x50, 0x61, 0x74, 0x63, 0x68, 0x42, 0x61, 0x79, 0xA1, 0x02, 0x0C, 0x00, 0xA4, 0x02, 0x0C,
						0x00, 0xA3, 0x03, 0x01, 0x01, 0xFF,
					},
				),
			},
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "R3LAYVirtualPatchBay",
				}: &Element{
					Path:        "1",
					ElementType: asn1.NodeType,
					Identifier:  "R3LAYVirtualPatchBay",
					IsOnline:    true,
				},
			},
			false,
		},
		{
			"+children",
			ElementCollection{},
			args{
				asn1.NewDecoder(
					[]byte{
						0x60, 0x80, 0x6B, 0x80, 0xA0, 0x80, 0x6A, 0x80, 0xA0, 0x07, 0x0D, 0x05, 0x01, 0x01, 0x02, 0x04,
						0x03, 0xA2, 0x80, 0x64, 0x80, 0xA0, 0x80, 0x61, 0x80, 0xA0, 0x03, 0x02, 0x01, 0x01, 0xA1, 0x80,
						0x31, 0x80, 0xA0, 0x04, 0x0C, 0x02, 0x4F, 0x6E, 0xAD, 0x03, 0x02, 0x01, 0x04, 0xA2, 0x03, 0x01,
						0x01, 0x00, 0xA5, 0x03, 0x02, 0x01, 0x03, 0xA9, 0x03, 0x01, 0x01, 0xFF, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0xA0, 0x80, 0x61, 0x80, 0xA0, 0x03, 0x02, 0x01, 0x02, 0xA1, 0x80, 0x31,
						0x80, 0xA0, 0x0F, 0x0C, 0x0D, 0x43, 0x6F, 0x72, 0x72, 0x20, 0x47, 0x61, 0x69, 0x6E, 0x5B, 0x64,
						0x42, 0x5D, 0xAD, 0x03, 0x02, 0x01, 0x01, 0xA2, 0x03, 0x02, 0x01, 0x00, 0xA5, 0x03, 0x02, 0x01,
						0x03, 0xA9, 0x03, 0x01, 0x01, 0xFF, 0xA4, 0x03, 0x02, 0x01, 0x00, 0xA3, 0x03, 0x02, 0x01, 0xF4,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA0, 0x80, 0x63, 0x80, 0xA0, 0x03, 0x02, 0x01,
						0x03, 0xA1, 0x80, 0x31, 0x80, 0xA0, 0x0C, 0x0C, 0x0A, 0x43, 0x6F, 0x6D, 0x70, 0x72, 0x65, 0x73,
						0x73, 0x6F, 0x72, 0xA3, 0x03, 0x01, 0x01, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0xA0, 0x80, 0x63, 0x80, 0xA0, 0x03, 0x02, 0x01, 0x04, 0xA1, 0x80, 0x31, 0x80, 0xA0, 0x0A, 0x0C,
						0x08, 0x45, 0x78, 0x70, 0x61, 0x6E, 0x64, 0x65, 0x72, 0xA3, 0x03, 0x01, 0x01, 0xFF, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA0, 0x80, 0x63, 0x80, 0xA0, 0x03, 0x02, 0x01, 0x05, 0xA1,
						0x80, 0x31, 0x80, 0xA0, 0x06, 0x0C, 0x04, 0x47, 0x61, 0x74, 0x65, 0xA3, 0x03, 0x01, 0x01, 0xFF,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA0, 0x80, 0x63, 0x80, 0xA0, 0x03, 0x02, 0x01,
						0x06, 0xA1, 0x80, 0x31, 0x80, 0xA0, 0x09, 0x0C, 0x07, 0x44, 0x65, 0x45, 0x73, 0x73, 0x65, 0x72,
						0xA3, 0x03, 0x01, 0x01, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					},
				),
			},
			ElementCollection{
				ElementKey{
					Path: "1.1.2.4.3",
				}: &Element{
					Path:        "1.1.2.4.3",
					ElementType: asn1.QualifiedNodeType,
					Children: []*Element{
						{
							Path:        "1",
							ElementType: "parameter",
							Identifier:  "On",
							IsOnline:    true,
							Access:      3,
							ValueType:   4,
							Value:       false,
						},
						{
							Path:        "2",
							ElementType: "parameter",
							Identifier:  "Corr Gain[dB]",
							Maximum:     int64(0),
							Minimum:     int64(-12),
							Value:       0,
							IsOnline:    true,
							Access:      3,
							ValueType:   1,
						},
						{
							Path:        "3",
							ElementType: "node",
							Identifier:  "Compressor",
							IsOnline:    true,
						},
						{
							Path:        "4",
							ElementType: "node",
							Identifier:  "Expander",
							IsOnline:    true,
						},
						{
							Path:        "5",
							ElementType: "node",
							Identifier:  "Gate",
							IsOnline:    true,
						},
						{
							Path:        "6",
							ElementType: "node",
							Identifier:  "DeEsser",
							IsOnline:    true,
						},
					},
				},
			},
			false,
		},
		{
			"+nodes",
			ElementCollection{},
			args{
				asn1.NewDecoder(
					[]byte{
						0x60, 0x82, 0x01, 0xFD, 0x6B, 0x82, 0x01, 0xF9, 0xA0, 0x40, 0x6A, 0x3E, 0xA0, 0x04, 0x0D, 0x02,
						0x01, 0x00, 0xA1, 0x36, 0x31, 0x34, 0xA0, 0x0A, 0x0C, 0x08, 0x69, 0x64, 0x65, 0x6E, 0x74, 0x69,
						0x74, 0x79, 0xA1, 0x02, 0x0C, 0x00, 0xA4, 0x1D, 0x0C, 0x1B, 0x64, 0x65, 0x2E, 0x6C, 0x2D, 0x73,
						0x2D, 0x62, 0x2E, 0x65, 0x6D, 0x62, 0x65, 0x72, 0x70, 0x6C, 0x75, 0x73, 0x2E, 0x69, 0x64, 0x65,
						0x6E, 0x74, 0x69, 0x74, 0x79, 0xA3, 0x03, 0x01, 0x01, 0xFF, 0xA0, 0x29, 0x6A, 0x27, 0xA0, 0x04,
						0x0D, 0x02, 0x01, 0x01, 0xA1, 0x1F, 0x31, 0x1D, 0xA0, 0x0E, 0x0C, 0x0C, 0x45, 0x6E, 0x76, 0x69,
						0x72, 0x6F, 0x6E, 0x6D, 0x65, 0x6E, 0x74, 0x73, 0xA1, 0x02, 0x0C, 0x00, 0xA4, 0x02, 0x0C, 0x00,
						0xA3, 0x03, 0x01, 0x01, 0xFF, 0xA0, 0x25, 0x6A, 0x23, 0xA0, 0x04, 0x0D, 0x02, 0x01, 0x02, 0xA1,
						0x1B, 0x31, 0x19, 0xA0, 0x0A, 0x0C, 0x08, 0x46, 0x75, 0x6E, 0x63, 0x74, 0x69, 0x6F, 0x6E, 0xA1,
						0x02, 0x0C, 0x00, 0xA4, 0x02, 0x0C, 0x00, 0xA3, 0x03, 0x01, 0x01, 0xFF, 0xA0, 0x24, 0x6A, 0x22,
						0xA0, 0x04, 0x0D, 0x02, 0x01, 0x03, 0xA1, 0x1A, 0x31, 0x18, 0xA0, 0x09, 0x0C, 0x07, 0x53, 0x6F,
						0x75, 0x72, 0x63, 0x65, 0x73, 0xA1, 0x02, 0x0C, 0x00, 0xA4, 0x02, 0x0C, 0x00, 0xA3, 0x03, 0x01,
						0x01, 0xFF, 0xA0, 0x22, 0x6A, 0x20, 0xA0, 0x04, 0x0D, 0x02, 0x01, 0x04, 0xA1, 0x18, 0x31, 0x16,
						0xA0, 0x07, 0x0C, 0x05, 0x53, 0x69, 0x6E, 0x6B, 0x73, 0xA1, 0x02, 0x0C, 0x00, 0xA4, 0x02, 0x0C,
						0x00, 0xA3, 0x03, 0x01, 0x01, 0xFF, 0xA0, 0x28, 0x6A, 0x26, 0xA0, 0x04, 0x0D, 0x02, 0x01, 0x05,
						0xA1, 0x1E, 0x31, 0x1C, 0xA0, 0x0D, 0x0C, 0x0B, 0x43, 0x6F, 0x6E, 0x6E, 0x65, 0x63, 0x74, 0x69,
						0x6F, 0x6E, 0x73, 0xA1, 0x02, 0x0C, 0x00, 0xA4, 0x02, 0x0C, 0x00, 0xA3, 0x03, 0x01, 0x01, 0xFF,
						0xA0, 0x2D, 0x6A, 0x2B, 0xA0, 0x04, 0x0D, 0x02, 0x01, 0x06, 0xA1, 0x23, 0x31, 0x21, 0xA0, 0x12,
						0x0C, 0x10, 0x53, 0x69, 0x6E, 0x6B, 0x53, 0x6F, 0x75, 0x72, 0x63, 0x65, 0x4D, 0x61, 0x74, 0x72,
						0x69, 0x78, 0xA1, 0x02, 0x0C, 0x00, 0xA4, 0x02, 0x0C, 0x00, 0xA3, 0x03, 0x01, 0x01, 0xFF, 0xA0,
						0x29, 0x6A, 0x27, 0xA0, 0x04, 0x0D, 0x02, 0x01, 0x07, 0xA1, 0x1F, 0x31, 0x1D, 0xA0, 0x0E, 0x0C,
						0x0C, 0x41, 0x75, 0x64, 0x69, 0x6F, 0x46, 0x69, 0x6C, 0x74, 0x65, 0x72, 0x73, 0xA1, 0x02, 0x0C,
						0x00, 0xA4, 0x02, 0x0C, 0x00, 0xA3, 0x03, 0x01, 0x01, 0xFF, 0xA0, 0x26, 0x6A, 0x24, 0xA0, 0x04,
						0x0D, 0x02, 0x01, 0x08, 0xA1, 0x1C, 0x31, 0x1A, 0xA0, 0x0B, 0x0C, 0x09, 0x53, 0x74, 0x72, 0x65,
						0x61, 0x6D, 0x69, 0x6E, 0x67, 0xA1, 0x02, 0x0C, 0x00, 0xA4, 0x02, 0x0C, 0x00, 0xA3, 0x03, 0x01,
						0x01, 0xFF, 0xA0, 0x24, 0x6A, 0x22, 0xA0, 0x04, 0x0D, 0x02, 0x01, 0x09, 0xA1, 0x1A, 0x31, 0x18,
						0xA0, 0x09, 0x0C, 0x07, 0x4C, 0x6F, 0x67, 0x69, 0x63, 0x61, 0x6C, 0xA1, 0x02, 0x0C, 0x00, 0xA4,
						0x02, 0x0C, 0x00, 0xA3, 0x03, 0x01, 0x01, 0xFF, 0xA0, 0x21, 0x6A, 0x1F, 0xA0, 0x04, 0x0D, 0x02,
						0x01, 0x0B, 0xA1, 0x17, 0x31, 0x15, 0xA0, 0x06, 0x0C, 0x04, 0x47, 0x50, 0x49, 0x4F, 0xA1, 0x02,
						0x0C, 0x00, 0xA4, 0x02, 0x0C, 0x00, 0xA3, 0x03, 0x01, 0x01, 0xFF, 0xA0, 0x24, 0x6A, 0x22, 0xA0,
						0x04, 0x0D, 0x02, 0x01, 0x0F, 0xA1, 0x1A, 0x31, 0x18, 0xA0, 0x09, 0x0C, 0x07, 0x4C, 0x69, 0x63,
						0x65, 0x6E, 0x73, 0x65, 0xA1, 0x02, 0x0C, 0x00, 0xA4, 0x02, 0x0C, 0x00, 0xA3, 0x03, 0x01, 0x01,
						0xFF,
					},
				),
			},
			ElementCollection{
				ElementKey{
					Path: "1.7",
					ID:   "AudioFilters",
				}: &Element{
					Path:        "1.7",
					ElementType: asn1.QualifiedNodeType,
					Identifier:  "AudioFilters",
					IsOnline:    true,
				},
				ElementKey{
					Path: "1.5",
					ID:   "Connections",
				}: &Element{
					Path:        "1.5",
					ElementType: asn1.QualifiedNodeType,
					Identifier:  "Connections",
					IsOnline:    true,
				},
				ElementKey{
					Path: "1.1",
					ID:   "Environments",
				}: &Element{
					Path:        "1.1",
					ElementType: asn1.QualifiedNodeType,
					Identifier:  "Environments",
					IsOnline:    true,
				},
				ElementKey{
					Path: "1.2",
					ID:   "Function",
				}: &Element{
					Path:        "1.2",
					ElementType: asn1.QualifiedNodeType,
					Identifier:  "Function",
					IsOnline:    true,
				},
				ElementKey{
					Path: "1.6",
					ID:   "SinkSourceMatrix",
				}: &Element{
					Path:        "1.6",
					ElementType: asn1.QualifiedNodeType,
					Identifier:  "SinkSourceMatrix",
					IsOnline:    true,
				},
				ElementKey{
					Path: "1.11",
					ID:   "GPIO",
				}: &Element{
					Path:        "1.11",
					ElementType: asn1.QualifiedNodeType,
					Identifier:  "GPIO",
					IsOnline:    true,
				},
				ElementKey{
					Path: "1.15",
					ID:   "License",
				}: &Element{
					Path:        "1.15",
					ElementType: asn1.QualifiedNodeType,
					Identifier:  "License",
					IsOnline:    true,
				},
				ElementKey{
					Path: "1.9",
					ID:   "Logical",
				}: &Element{
					Path:        "1.9",
					ElementType: asn1.QualifiedNodeType,
					Identifier:  "Logical",
					IsOnline:    true,
				},
				ElementKey{
					Path: "1.4",
					ID:   "Sinks",
				}: &Element{
					Path:        "1.4",
					ElementType: asn1.QualifiedNodeType,
					Identifier:  "Sinks",
					IsOnline:    true,
				},
				ElementKey{
					Path: "1.3",
					ID:   "Sources",
				}: &Element{
					Path:        "1.3",
					ElementType: asn1.QualifiedNodeType,
					Identifier:  "Sources",
					IsOnline:    true,
				},
				ElementKey{
					Path: "1.8",
					ID:   "Streaming",
				}: &Element{
					Path:        "1.8",
					ElementType: asn1.QualifiedNodeType,
					Identifier:  "Streaming",
					IsOnline:    true,
				},
				ElementKey{
					Path: "1.0",
					ID:   "identity",
				}: &Element{
					Path:        "1.0",
					ElementType: asn1.QualifiedNodeType,
					Identifier:  "identity",
					IsOnline:    true,
				},
			},
			false,
		},
		{
			"+function",
			ElementCollection{},
			args{
				asn1.NewDecoder(
					[]byte{
						0x60, 0x82, 0x01, 0x01, 0x6B, 0x82, 0x00, 0xFD, 0xA0, 0x23, 0x69, 0x21, 0xA0, 0x05, 0x0D, 0x03,
						0x01, 0x02, 0x01, 0xA1, 0x18, 0x31, 0x16, 0xA0, 0x06, 0x0C, 0x04, 0x43, 0x61, 0x6C, 0x6C, 0xAD,
						0x03, 0x02, 0x01, 0x03, 0xA5, 0x03, 0x02, 0x01, 0x03, 0xA2, 0x02, 0x0C, 0x00, 0xA0, 0x82, 0x00,
						0x81, 0x69, 0x7F, 0xA0, 0x05, 0x0D, 0x03, 0x01, 0x02, 0x02, 0xA1, 0x76, 0x31, 0x74, 0xA0, 0x08,
						0x0C, 0x06, 0x52, 0x65, 0x74, 0x75, 0x72, 0x6E, 0xAD, 0x03, 0x02, 0x01, 0x03, 0xA5, 0x03, 0x02,
						0x01, 0x03, 0xA2, 0x5E, 0x0C, 0x5C, 0x5B, 0x7B, 0x22, 0x66, 0x6E, 0x22, 0x3A, 0x22, 0x63, 0x6F,
						0x6E, 0x6E, 0x65, 0x63, 0x74, 0x74, 0x6F, 0x67, 0x75, 0x69, 0x22, 0x2C, 0x22, 0x66, 0x6E, 0x49,
						0x44, 0x22, 0x3A, 0x22, 0x34, 0x35, 0x30, 0x34, 0x22, 0x2C, 0x22, 0x66, 0x6E, 0x52, 0x65, 0x74,
						0x54, 0x79, 0x70, 0x65, 0x22, 0x3A, 0x22, 0x4A, 0x53, 0x4F, 0x4E, 0x22, 0x2C, 0x22, 0x66, 0x6E,
						0x52, 0x65, 0x74, 0x56, 0x61, 0x6C, 0x75, 0x65, 0x22, 0x3A, 0x5B, 0x22, 0x73, 0x74, 0x61, 0x74,
						0x65, 0x22, 0x3A, 0x22, 0x63, 0x6F, 0x6E, 0x6E, 0x65, 0x63, 0x74, 0x69, 0x6E, 0x67, 0x22, 0x5D,
						0x7D, 0x5D, 0xA0, 0x51, 0x74, 0x4F, 0xA0, 0x05, 0x0D, 0x03, 0x01, 0x02, 0x03, 0xA1, 0x46, 0x31,
						0x44, 0xA0, 0x0E, 0x0C, 0x0C, 0x43, 0x61, 0x6C, 0x6C, 0x46, 0x72, 0x6F, 0x6D, 0x4A, 0x53, 0x4F,
						0x4E, 0xA2, 0x17, 0x30, 0x15, 0xA0, 0x13, 0x75, 0x11, 0xA0, 0x03, 0x02, 0x01, 0x03, 0xA1, 0x0A,
						0x0C, 0x08, 0x70, 0x61, 0x72, 0x73, 0x4A, 0x53, 0x4F, 0x4E, 0xA3, 0x19, 0x30, 0x17, 0xA0, 0x15,
						0x75, 0x13, 0xA0, 0x03, 0x02, 0x01, 0x03, 0xA1, 0x0C, 0x0C, 0x0A, 0x72, 0x65, 0x73, 0x75, 0x6C,
						0x74, 0x4A, 0x53, 0x4F, 0x4E,
					},
				),
			},
			ElementCollection{
				ElementKey{
					Path: "1.2.1",
					ID:   "Call",
				}: &Element{
					Path:        "1.2.1",
					ElementType: asn1.QualifiedParameterType,
					Identifier:  "Call",
					Value:       "",
					Access:      3,
					ValueType:   3,
				},
				ElementKey{
					Path: "1.2.2",
					ID:   "Return",
				}: &Element{
					Path:        "1.2.2",
					ElementType: asn1.QualifiedParameterType,
					Identifier:  "Return",
					Value: string(
						`[{"fn":"connecttogui","fnID":"4504","fnRetType":"JSON","fnRetValue":["state":"connecting"]}]`,
					),
					Access:    3,
					ValueType: 3,
				},
				ElementKey{
					Path: "1.2.3",
					ID:   "CallFromJSON",
				}: &Element{
					Path:        "1.2.3",
					ElementType: asn1.FunctionType,
					Identifier:  "CallFromJSON",
				}},
			false,
		},
		{
			"+children2",
			ElementCollection{},
			args{
				asn1.NewDecoder(
					[]byte{
						0x60, 0x80, 0x6B, 0x80, 0xA0, 0x80, 0x6A, 0x80, 0xA0, 0x07, 0x0D, 0x05, 0x01, 0x01, 0x02, 0x04,
						0x02, 0xA2, 0x80, 0x64, 0x80, 0xA0, 0x80, 0x61, 0x80, 0xA0, 0x03, 0x02, 0x01, 0x01, 0xA1, 0x80,
						0x31, 0x80, 0xA0, 0x04, 0x0C, 0x02, 0x4F, 0x6E, 0xAD, 0x03, 0x02, 0x01, 0x04, 0xA2, 0x03, 0x01,
						0x01, 0x00, 0xA5, 0x03, 0x02, 0x01, 0x03, 0xA9, 0x03, 0x01, 0x01, 0xFF, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0xA0, 0x80, 0x63, 0x80, 0xA0, 0x03, 0x02, 0x01, 0x02, 0xA1, 0x80, 0x31,
						0x80, 0xA0, 0x08, 0x0C, 0x06, 0x42, 0x61, 0x6E, 0x64, 0x20, 0x31, 0xA3, 0x03, 0x01, 0x01, 0xFF,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA0, 0x80, 0x63, 0x80, 0xA0, 0x03, 0x02, 0x01,
						0x03, 0xA1, 0x80, 0x31, 0x80, 0xA0, 0x08, 0x0C, 0x06, 0x42, 0x61, 0x6E, 0x64, 0x20, 0x32, 0xA3,
						0x03, 0x01, 0x01, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA0, 0x80, 0x63, 0x80,
						0xA0, 0x03, 0x02, 0x01, 0x04, 0xA1, 0x80, 0x31, 0x80, 0xA0, 0x08, 0x0C, 0x06, 0x42, 0x61, 0x6E,
						0x64, 0x20, 0x33, 0xA3, 0x03, 0x01, 0x01, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0xA0, 0x80, 0x63, 0x80, 0xA0, 0x03, 0x02, 0x01, 0x05, 0xA1, 0x80, 0x31, 0x80, 0xA0, 0x08, 0x0C,
						0x06, 0x42, 0x61, 0x6E, 0x64, 0x20, 0x34, 0xA3, 0x03, 0x01, 0x01, 0xFF, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0xA0, 0x80, 0x63, 0x80, 0xA0, 0x03, 0x02, 0x01, 0x06, 0xA1, 0x80, 0x31,
						0x80, 0xA0, 0x08, 0x0C, 0x06, 0x42, 0x61, 0x6E, 0x64, 0x20, 0x35, 0xA3, 0x03, 0x01, 0x01, 0xFF,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00,
					},
				),
			},
			ElementCollection{
				ElementKey{
					Path: "1.1.2.4.2",
				}: &Element{
					Path:        "1.1.2.4.2",
					ElementType: asn1.QualifiedNodeType,
					Children: []*Element{
						{
							Path:        "1",
							ElementType: "parameter",
							Identifier:  "On",
							IsOnline:    true,
							Access:      3,
							ValueType:   4,
							Value:       false,
						},
						{
							Path:        "2",
							ElementType: "node",
							Identifier:  "Band 1",
							IsOnline:    true,
						},
						{
							Path:        "3",
							ElementType: "node",
							Identifier:  "Band 2",
							IsOnline:    true,
						},
						{
							Path:        "4",
							ElementType: "node",
							Identifier:  "Band 3",
							IsOnline:    true,
						},
						{
							Path:        "5",
							ElementType: "node",
							Identifier:  "Band 4",
							IsOnline:    true,
						},
						{
							Path:        "6",
							ElementType: "node",
							Identifier:  "Band 5",
							IsOnline:    true,
						},
					},
				},
			},
			false,
		},
		{
			"+withEnumeration",
			ElementCollection{},
			args{
				asn1.NewDecoder(
					[]byte{
						0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x6a, 0x80, 0xa0, 0x05, 0x0d, 0x03, 0x01, 0x1d, 0x03, 0xa2,
						0x80, 0x64, 0x80, 0xa0, 0x80, 0x61, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80,
						0xa0, 0x0f, 0x0c, 0x0d, 0x50, 0x72, 0x69, 0x6f, 0x20, 0x53, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x63,
						0x65, 0xad, 0x03, 0x02, 0x01, 0x06, 0xa7, 0x82, 0x00, 0x8d, 0x0c, 0x82, 0x00, 0x89, 0x50, 0x54,
						0x50, 0x20, 0x2d, 0x20, 0x4d, 0x41, 0x44, 0x49, 0x20, 0x2d, 0x20, 0x57, 0x6f, 0x72, 0x64, 0x43,
						0x6c, 0x6f, 0x63, 0x6b, 0x0a, 0x50, 0x54, 0x50, 0x20, 0x2d, 0x20, 0x57, 0x6f, 0x72, 0x64, 0x43,
						0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x2d, 0x20, 0x4d, 0x41, 0x44, 0x49, 0x0a, 0x4d, 0x41, 0x44, 0x49,
						0x20, 0x2d, 0x20, 0x50, 0x54, 0x50, 0x20, 0x2d, 0x20, 0x57, 0x6f, 0x72, 0x64, 0x43, 0x6c, 0x6f,
						0x63, 0x6b, 0x0a, 0x4d, 0x41, 0x44, 0x49, 0x20, 0x2d, 0x20, 0x57, 0x6f, 0x72, 0x64, 0x43, 0x6c,
						0x6f, 0x63, 0x6b, 0x20, 0x2d, 0x20, 0x50, 0x54, 0x50, 0x0a, 0x57, 0x6f, 0x72, 0x64, 0x43, 0x6c,
						0x6f, 0x63, 0x6b, 0x20, 0x2d, 0x20, 0x50, 0x54, 0x50, 0x20, 0x2d, 0x20, 0x4d, 0x41, 0x44, 0x49,
						0x0a, 0x57, 0x6f, 0x72, 0x64, 0x43, 0x6c, 0x6f, 0x63, 0x6b, 0x20, 0x2d, 0x20, 0x4d, 0x41, 0x44,
						0x49, 0x20, 0x2d, 0x20, 0x50, 0x54, 0x50, 0xa2, 0x03, 0x02, 0x01, 0x01, 0xa5, 0x03, 0x02, 0x01,
						0x01, 0xa9, 0x03, 0x01, 0x01, 0xff, 0xa4, 0x03, 0x02, 0x01, 0x05, 0xa3, 0x03, 0x02, 0x01, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01,
						0x02, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x05, 0x0c, 0x03, 0x50, 0x54, 0x50, 0xa3, 0x03, 0x01, 0x01,
						0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x63, 0x80, 0xa0, 0x03, 0x02,
						0x01, 0x03, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c, 0x04, 0x4d, 0x41, 0x44, 0x49, 0xa3, 0x03,
						0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x63, 0x80, 0xa0,
						0x03, 0x02, 0x01, 0x04, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x0b, 0x0c, 0x09, 0x57, 0x6f, 0x72, 0x64,
						0x43, 0x6c, 0x6f, 0x63, 0x6b, 0xa3, 0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0xa0, 0x80, 0x63, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x05, 0xa1, 0x80, 0x31, 0x80, 0xa0,
						0x0a, 0x0c, 0x08, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x61, 0x6c, 0xa3, 0x03, 0x01, 0x01, 0xff,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00,
					},
				),
			},
			ElementCollection{
				ElementKey{
					Path: "1.29.3",
				}: &Element{
					Path:        "1.29.3",
					ElementType: asn1.QualifiedNodeType,
					Children: []*Element{
						{
							Path:        "1",
							ElementType: "parameter",
							Identifier:  "Prio Sequence",
							IsOnline:    true,
							Maximum:     int64(5),
							Minimum:     int64(0),
							Value:       1,
							Access:      1,
							//nolint:dupword
							Enumeration: "PTP - MADI - WordClock\nPTP - WordClock - MADI\nMADI - PTP - WordClock\n" +
								"MADI - WordClock - PTP\nWordClock - PTP - MADI\nWordClock - MADI - PTP",
							ValueType: 6,
						},
						{
							Path:        "2",
							ElementType: "node",
							Identifier:  "PTP",
							IsOnline:    true,
						},
						{
							Path:        "3",
							ElementType: "node",
							Identifier:  "MADI",
							IsOnline:    true,
						},
						{
							Path:        "4",
							ElementType: "node",
							Identifier:  "WordClock",
							IsOnline:    true,
						},
						{
							Path:        "5",
							ElementType: "node",
							Identifier:  "Internal",
							IsOnline:    true,
						},
					},
				},
			},
			false,
		},
		{
			"+multipleElements",
			ElementCollection{},
			args{
				asn1.NewDecoder(
					[]byte{
						0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x6a, 0x80, 0xa0, 0x03, 0x0d, 0x01, 0x01, 0xa1, 0x80, 0x31,
						0x80, 0xa0, 0x0c, 0x0c, 0x0a, 0x56, 0x69, 0x73, 0x54, 0x6f, 0x6f, 0x6c, 0x4d, 0x6b, 0x32, 0xa2,
						0x03, 0x01, 0x01, 0xff, 0xa3, 0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0xa0, 0x80, 0x6a, 0x80, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x01, 0xa1, 0x80, 0x31, 0x80, 0xa0,
						0x0c, 0x0c, 0x0a, 0x4d, 0x61, 0x6e, 0x61, 0x67, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0xa2, 0x03, 0x01,
						0x01, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0,
						0x80, 0x6a, 0x80, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x02, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x06, 0x0c,
						0x04, 0x49, 0x6e, 0x66, 0x6f, 0xa2, 0x03, 0x01, 0x01, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xff, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa0, 0x80, 0x6a, 0x80, 0xa0, 0x04, 0x0d, 0x02, 0x01,
						0x03, 0xa1, 0x80, 0x31, 0x80, 0xa0, 0x09, 0x0c, 0x07, 0x50, 0x72, 0x6f, 0x6a, 0x65, 0x63, 0x74,
						0xa2, 0x03, 0x01, 0x01, 0x00, 0xa3, 0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0xa0, 0x80, 0x6a, 0x80, 0xa0, 0x04, 0x0d, 0x02, 0x01, 0x1e, 0xa1, 0x80, 0x31, 0x80,
						0xa0, 0x0a, 0x0c, 0x08, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0xa2, 0x03, 0x01, 0x01,
						0x00, 0xa3, 0x03, 0x01, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00,
					},
				),
			},
			ElementCollection{
				ElementKey{
					Path: "1.2",
					ID:   "Info",
				}: &Element{
					Path:        "1.2",
					ElementType: asn1.QualifiedNodeType,
					Identifier:  "Info",
					IsOnline:    true,
				},
				ElementKey{
					Path: "1.1",
					ID:   "Management",
				}: &Element{
					Path:        "1.1",
					ElementType: asn1.QualifiedNodeType,
					Identifier:  "Management",
					IsOnline:    true,
				},
				ElementKey{
					Path: "1.3",
					ID:   "Project",
				}: &Element{
					Path:        "1.3",
					ElementType: asn1.QualifiedNodeType,
					Identifier:  "Project",
					IsOnline:    true,
				},
				ElementKey{
					Path: "1",
					ID:   "VisToolMk2",
				}: &Element{
					Path:        "1",
					ElementType: asn1.QualifiedNodeType,
					Identifier:  "VisToolMk2",
					IsOnline:    true,
					IsRoot:      true,
				},
				ElementKey{
					Path: "1.30",
					ID:   "identity",
				}: &Element{
					Path:        "1.30",
					ElementType: asn1.QualifiedNodeType,
					Identifier:  "identity",
					IsOnline:    true,
				},
			},
			false,
		},
		{
			"-elementContextEndErr",
			ElementCollection{},
			args{
				asn1.NewDecoder(
					[]byte{
						0x60, 0x80, 0x6B, 0x80, 0xA0, 0x80, 0x63, 0x80, 0xA0, 0x03, 0x02, 0x01, 0x01, 0xA1, 0x27, 0x31,
						0x25, 0xA0, 0x16, 0x0C, 0x14, 0x52, 0x33, 0x4C, 0x41, 0x59, 0x56, 0x69, 0x72, 0x74, 0x75, 0x61,
						0x6C, 0x50, 0x61, 0x74, 0x63, 0x68, 0x42, 0x61, 0x79, 0xA1, 0x02, 0x0C, 0x00, 0xA4, 0x02, 0x0C,
						0x00, 0xA3, 0x03, 0x01, 0x01, 0xFF, 0x00, 0x00, 0x00,
					},
				),
			},
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "R3LAYVirtualPatchBay",
				}: &Element{
					Path:        "1",
					ElementType: asn1.NodeType,
					Identifier:  "R3LAYVirtualPatchBay",
					IsOnline:    true,
				},
			},
			true,
		},
		{
			"-elementEndErr",
			ElementCollection{},
			args{
				asn1.NewDecoder(
					[]byte{
						0x60, 0x80, 0x6B, 0x80, 0xA0, 0x80, 0x63, 0x80, 0xA0, 0x03, 0x02, 0x01, 0x01, 0xA1, 0x27, 0x31,
						0x25, 0xA0, 0x16, 0x0C, 0x14, 0x52, 0x33, 0x4C, 0x41, 0x59, 0x56, 0x69, 0x72, 0x74, 0x75, 0x61,
						0x6C, 0x50, 0x61, 0x74, 0x63, 0x68, 0x42, 0x61, 0x79, 0xA1, 0x02, 0x0C, 0x00, 0xA4, 0x02, 0x0C,
						0x00, 0xA3, 0x03, 0x01, 0x01, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00,
					},
				),
			},
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "R3LAYVirtualPatchBay",
				}: &Element{
					Path:        "1",
					ElementType: asn1.NodeType,
					Identifier:  "R3LAYVirtualPatchBay",
					IsOnline:    true,
				},
			},
			true,
		},
		{
			"-applicationEndErr",
			ElementCollection{},
			args{
				asn1.NewDecoder(
					[]byte{
						0x60, 0x35, 0x6B, 0x80, 0xA0, 0x30, 0x63, 0x2E, 0xA0, 0x03, 0x02, 0x01, 0x01, 0xA1, 0x27, 0x31,
						0x25, 0xA0, 0x16, 0x0C, 0x14, 0x52, 0x33, 0x4C, 0x41, 0x59, 0x56, 0x69, 0x72, 0x74, 0x75, 0x61,
						0x6C, 0x50, 0x61, 0x74, 0x63, 0x68, 0x42, 0x61, 0x79, 0xA1, 0x02, 0x0C, 0x00, 0xA4, 0x02, 0x0C,
						0x00, 0xA3, 0x03, 0x01, 0x01, 0xFF, 0x00,
					},
				),
			},
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "R3LAYVirtualPatchBay",
				}: &Element{
					Path:        "1",
					ElementType: asn1.NodeType,
					Identifier:  "R3LAYVirtualPatchBay",
					IsOnline:    true,
				},
			},
			true,
		},
		{
			"-collectionEndErr",
			ElementCollection{},
			args{
				asn1.NewDecoder(
					[]byte{
						0x60, 0x35, 0x6B, 0x32, 0xA0, 0x30, 0x63, 0x2E, 0xA0, 0x03, 0x02, 0x01, 0x01, 0xA1, 0x27, 0x31,
						0x25, 0xA0, 0x16, 0x0C, 0x14, 0x52, 0x33, 0x4C, 0x41, 0x59, 0x56, 0x69, 0x72, 0x74, 0x75, 0x61,
						0x6C, 0x50, 0x61, 0x74, 0x63, 0x68, 0x42, 0x61, 0x79, 0xA1, 0x02, 0x0C, 0x00, 0xA4, 0x02, 0x0C,
						0x00, 0xA3, 0x03, 0x01, 0x01, 0xFF, 0x00,
					},
				),
			},
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "R3LAYVirtualPatchBay",
				}: &Element{
					Path:        "1",
					ElementType: asn1.NodeType,
					Identifier:  "R3LAYVirtualPatchBay",
					IsOnline:    true,
				},
			},
			true,
		},
		{
			"-collectionStillHasData",
			ElementCollection{},
			args{
				asn1.NewDecoder(
					[]byte{
						0x60, 0x36, 0x6B, 0x32, 0xA0, 0x30, 0x63, 0x2E, 0xA0, 0x03, 0x02, 0x01, 0x01, 0xA1, 0x27, 0x31,
						0x25, 0xA0, 0x16, 0x0C, 0x14, 0x52, 0x33, 0x4C, 0x41, 0x59, 0x56, 0x69, 0x72, 0x74, 0x75, 0x61,
						0x6C, 0x50, 0x61, 0x74, 0x63, 0x68, 0x42, 0x61, 0x79, 0xA1, 0x02, 0x0C, 0x00, 0xA4, 0x02, 0x0C,
						0x00, 0xA3, 0x03, 0x01, 0x01, 0xFF, 0x01, 0x02,
					},
				),
			},
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "R3LAYVirtualPatchBay",
				}: &Element{
					Path:        "1",
					ElementType: asn1.NodeType,
					Identifier:  "R3LAYVirtualPatchBay",
					IsOnline:    true,
				},
			},
			true,
		},
		{
			"-invalidElement",
			ElementCollection{},
			args{
				asn1.NewDecoder(
					[]byte{
						0x60, 0x80, 0x6B, 0x80, 0xA0, 0x80, 0x63, 0x2E, 0xA0,
					},
				),
			},
			ElementCollection{},
			true,
		},
		{
			"-context0Invalid",
			ElementCollection{},
			args{
				asn1.NewDecoder([]byte{0x60, 0x80, 0x6B, 0x80, 0xA0, 0x30, 0x63}),
			},
			ElementCollection{},
			true,
		},
		{
			"-noRootElementCollection",
			ElementCollection{},
			args{asn1.NewDecoder([]byte{0x60, 0x80, 0x6B})},
			ElementCollection{},
			true,
		},
		{
			"-empty",
			ElementCollection{},
			args{asn1.NewDecoder([]byte{})},
			ElementCollection{},
			true,
		},
	}

	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			if err := tt.ec.Populate(tt.args.data); (err != nil) != tt.wantErr {
				t.Fatalf("ElementCollection.Populate() error = %v, wantErr %v", err, tt.wantErr)
			}

			if len(tt.want) != len(tt.ec) {
				t.Fatalf("ElementCollection.Populate() =expected len %d, go %d", len(tt.want), len(tt.ec))
			}

			if diff := cmp.Diff(tt.want, tt.ec); diff != "" {
				t.Fatalf("ElementCollection.Populate() = %s", diff)
			}
		})
	}
}

func TestElementCollection_GetElementByPath(t *testing.T) {
	t.Parallel()

	type args struct {
		currentPath string
	}

	tests := []struct {
		name    string
		ec      ElementCollection
		args    args
		want    *Element
		wantErr bool
	}{
		{
			"+valid",
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "test",
				}: &Element{
					Path:        "1",
					ElementType: "node",
					Identifier:  "test",
					Description: "foobar",
				},
			},
			args{
				"1",
			},
			&Element{
				Path:        "1",
				ElementType: "node",
				Identifier:  "test",
				Description: "foobar",
			},
			false,
		},
		{
			"+multiple",
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "test",
				}: &Element{
					Path:        "1",
					ElementType: "node",
					Identifier:  "test",
					Description: "foobar",
				},
				ElementKey{
					Path: "2",
					ID:   "test2",
				}: &Element{
					Path:        "2",
					ElementType: "node",
					Identifier:  "test2",
					Description: "foobar",
				},
			},
			args{
				"1",
			},
			&Element{
				Path:        "1",
				ElementType: "node",
				Identifier:  "test",
				Description: "foobar",
			},
			false,
		},
		{
			"+child",
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "foo",
				}: &Element{
					Path:        "1",
					ElementType: "node",
					Identifier:  "foo",
					Description: "foobar",
					Children: []*Element{
						{
							Path:        "2",
							ElementType: "node",
							Identifier:  "bar",
							Description: "foobar",
						},
					},
				},
			},
			args{
				"1.2",
			},
			&Element{
				Path:        "2",
				ElementType: "node",
				Identifier:  "bar",
				Description: "foobar",
			},
			false,
		},
		{
			"-notFound",
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "test",
				}: &Element{
					Path:        "1",
					ElementType: "node",
					Identifier:  "test",
					Description: "foobar",
				},
				ElementKey{
					Path: "2",
					ID:   "test2",
				}: &Element{
					Path:        "2",
					ElementType: "node",
					Identifier:  "test2",
					Description: "foobar",
				},
			},
			args{
				"12",
			},
			nil,
			true,
		},
	}

	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got, err := tt.ec.GetElementByPath(tt.args.currentPath)
			if (err != nil) != tt.wantErr {
				t.Fatalf("ElementCollection.GetElementByPath() error = %v, wantErr %v", err, tt.wantErr)
			}

			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("ElementCollection.GetElementByPath() = %s", diff)
			}
		})
	}
}

func TestElementCollection_GetElementByID(t *testing.T) {
	t.Parallel()

	type args struct {
		id string
	}

	tests := []struct {
		name     string
		ec       ElementCollection
		args     args
		wantEl   *Element
		wantPath string
		wantErr  bool
	}{
		{
			"+valid",
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "test",
				}: &Element{
					Path:        "1",
					ElementType: "node",
					Identifier:  "test",
					Description: "foobar",
				},
			},
			args{
				"test",
			},
			&Element{
				Path:        "1",
				ElementType: "node",
				Identifier:  "test",
				Description: "foobar",
			},
			"1",
			false,
		},
		{
			"+multiple",
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "test",
				}: &Element{
					Path:        "1",
					ElementType: "node",
					Identifier:  "test",
					Description: "foobar",
				},
				ElementKey{
					Path: "2",
					ID:   "test2",
				}: &Element{
					Path:        "2",
					ElementType: "node",
					Identifier:  "test2",
					Description: "foobar",
				},
			},
			args{
				"test",
			},
			&Element{
				Path:        "1",
				ElementType: "node",
				Identifier:  "test",
				Description: "foobar",
			},
			"1",
			false,
		},
		{
			"+child",
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "foo",
				}: &Element{
					Path:        "1",
					ElementType: "node",
					Identifier:  "foo",
					Description: "foobar",
					Children: []*Element{
						{
							Path:        "2",
							ElementType: "node",
							Identifier:  "bar",
							Description: "foobar",
						},
					},
				},
			},
			args{
				"bar",
			},
			&Element{
				Path:        "2",
				ElementType: "node",
				Identifier:  "bar",
				Description: "foobar",
			},
			"1.2",
			false,
		},
		{
			"-notFound",
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "test",
				}: &Element{
					Path:        "1",
					ElementType: "node",
					Identifier:  "test",
					Description: "foobar",
				},
				ElementKey{
					Path: "2",
					ID:   "test2",
				}: &Element{
					Path:        "2",
					ElementType: "node",
					Identifier:  "test2",
					Description: "foobar",
				},
			},
			args{
				"foobar",
			},
			nil,
			"",
			true,
		},
	}

	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got, got1, err := tt.ec.GetElementByID(tt.args.id)
			if (err != nil) != tt.wantErr {
				t.Fatalf("ElementCollection.GetElementByID() error = %v, wantErr %v", err, tt.wantErr)
			}

			if diff := cmp.Diff(tt.wantEl, got); diff != "" {
				t.Fatalf("ElementCollection.GetElementByID() element = %s", diff)
			}

			if diff := cmp.Diff(tt.wantPath, got1); diff != "" {
				t.Fatalf("ElementCollection.GetElementByID() path = %s", diff)
			}
		})
	}
}

func TestElementCollection_MarshalJSON(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name    string
		ec      ElementCollection
		want    []byte
		wantErr bool
	}{
		{
			"+node",
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "test",
				}: &Element{
					Path:        "1",
					ElementType: "node",
					Identifier:  "test",
					Description: "foobar",
				},
			},
			[]byte{
				0x7b, 0x22, 0x31, 0x22, 0x3a, 0x7b, 0x22, 0x70, 0x61, 0x74, 0x68, 0x22, 0x3a, 0x22, 0x31, 0x22, 0x2c,
				0x22, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x5f, 0x74, 0x79, 0x70, 0x65, 0x22, 0x3a, 0x22, 0x6e,
				0x6f, 0x64, 0x65, 0x22, 0x2c, 0x22, 0x63, 0x68, 0x69, 0x6c, 0x64, 0x72, 0x65, 0x6e, 0x22, 0x3a, 0x6e,
				0x75, 0x6c, 0x6c, 0x2c, 0x22, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x66, 0x69, 0x65, 0x72, 0x22, 0x3a,
				0x22, 0x74, 0x65, 0x73, 0x74, 0x22, 0x2c, 0x22, 0x64, 0x65, 0x73, 0x63, 0x72, 0x69, 0x70, 0x74, 0x69,
				0x6f, 0x6e, 0x22, 0x3a, 0x22, 0x66, 0x6f, 0x6f, 0x62, 0x61, 0x72, 0x22, 0x2c, 0x22, 0x69, 0x73, 0x5f,
				0x6f, 0x6e, 0x6c, 0x69, 0x6e, 0x65, 0x22, 0x3a, 0x66, 0x61, 0x6c, 0x73, 0x65, 0x2c, 0x22, 0x69, 0x73,
				0x5f, 0x72, 0x6f, 0x6f, 0x74, 0x22, 0x3a, 0x66, 0x61, 0x6c, 0x73, 0x65, 0x7d, 0x7d,
			},
			false,
		},
		{
			"+parameter",
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "test",
				}: &Element{
					Path:        "1",
					ElementType: "parameter",
					Identifier:  "test",
					Description: "foobar",
					Value:       true,
					ValueType:   4,
				},
			},
			[]byte{
				0x7b, 0x22, 0x31, 0x22, 0x3a, 0x7b, 0x22, 0x70, 0x61, 0x74, 0x68, 0x22, 0x3a, 0x22, 0x31, 0x22, 0x2c,
				0x22, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x5f, 0x74, 0x79, 0x70, 0x65, 0x22, 0x3a, 0x22, 0x70,
				0x61, 0x72, 0x61, 0x6d, 0x65, 0x74, 0x65, 0x72, 0x22, 0x2c, 0x22, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69,
				0x66, 0x69, 0x65, 0x72, 0x22, 0x3a, 0x22, 0x74, 0x65, 0x73, 0x74, 0x22, 0x2c, 0x22, 0x64, 0x65, 0x73,
				0x63, 0x72, 0x69, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0x22, 0x3a, 0x22, 0x66, 0x6f, 0x6f, 0x62, 0x61, 0x72,
				0x22, 0x2c, 0x22, 0x76, 0x61, 0x6c, 0x75, 0x65, 0x22, 0x3a, 0x74, 0x72, 0x75, 0x65, 0x2c, 0x22, 0x74,
				0x79, 0x70, 0x65, 0x22, 0x3a, 0x34, 0x7d, 0x7d,
			},
			false,
		},
		{
			"+function",
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "test",
				}: &Element{
					Path:        "1",
					ElementType: "function",
					Identifier:  "test",
					Description: "foobar",
				},
			},
			[]byte{
				0x7b, 0x22, 0x31, 0x22, 0x3a, 0x7b, 0x22, 0x70, 0x61, 0x74, 0x68, 0x22, 0x3a, 0x22, 0x31, 0x22,
				0x2c, 0x22, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x5f, 0x74, 0x79, 0x70, 0x65, 0x22, 0x3a,
				0x22, 0x66, 0x75, 0x6e, 0x63, 0x74, 0x69, 0x6f, 0x6e, 0x22, 0x2c, 0x22, 0x63, 0x68, 0x69, 0x6c,
				0x64, 0x72, 0x65, 0x6e, 0x22, 0x3a, 0x6e, 0x75, 0x6c, 0x6c, 0x2c, 0x22, 0x69, 0x64, 0x65, 0x6e,
				0x74, 0x69, 0x66, 0x69, 0x65, 0x72, 0x22, 0x3a, 0x22, 0x74, 0x65, 0x73, 0x74, 0x22, 0x2c, 0x22,
				0x64, 0x65, 0x73, 0x63, 0x72, 0x69, 0x70, 0x74, 0x69, 0x6f, 0x6e, 0x22, 0x3a, 0x22, 0x66, 0x6f,
				0x6f, 0x62, 0x61, 0x72, 0x22, 0x7d, 0x7d},
			false,
		},
		{
			"+empty",
			ElementCollection{},
			[]byte{0x7b, 0x7d},
			false,
		},
		{
			"-unknownType",
			ElementCollection{
				ElementKey{
					Path: "1",
					ID:   "test",
				}: &Element{
					Path:        "1",
					ElementType: "foobar",
					Identifier:  "test",
					Description: "foobar",
				},
			},
			nil,
			true,
		},
	}
	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got, err := tt.ec.MarshalJSON()
			if (err != nil) != tt.wantErr {
				t.Fatalf("ElementCollection.MarshalJSON() error = %v, wantErr %v", err, tt.wantErr)
			}

			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("ElementCollection.MarshalJSON() = %s", diff)
			}
		})
	}
}

func TestNewElementConnection(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name string
		want ElementCollection
	}{
		{
			"+valid",
			make(ElementCollection),
		},
	}
	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got := NewElementConnection()
			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("NewElementConnection() = %s", diff)
			}
		})
	}
}

func TestGetRootRequest(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name    string
		want    []byte
		wantErr bool
	}{
		{
			"+valid",
			[]byte{
				0xfe, 0x00, 0x0e, 0x00, 0x01, 0x80, 0x01, 0x02, 0x28, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80,
				0x62, 0x80, 0xa0, 0x03, 0x02, 0x01, 0x20, 0xa1, 0x03, 0x02, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x30, 0xff, 0xfe, 0x00, 0x0e, 0x00, 0x01, 0x40, 0x01, 0x02,
				0x28, 0x02, 0x01, 0x23, 0xff,
			},
			false,
		},
	}
	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got, err := GetRootRequest()
			if (err != nil) != tt.wantErr {
				t.Fatalf("GetRootRequest() error = %v, wantErr %v", err, tt.wantErr)
			}

			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("GetRootRequest() = %s", diff)
			}
		})
	}
}

func TestGetRequestByType(t *testing.T) {
	t.Parallel()

	type args struct {
		et   ElementType
		path string
	}

	tests := []struct {
		name    string
		args    args
		want    []byte
		wantErr bool
	}{
		{
			"+valid",
			args{
				"node",
				"1",
			},
			[]byte{
				0xfe, 0x00, 0x0e, 0x00, 0x01, 0x80, 0x01, 0x02, 0x28, 0x02, 0x60, 0x80, 0x6b, 0x80, 0xa0, 0x80, 0x6a,
				0x80, 0xa0, 0x80, 0x0d, 0x01, 0x01, 0xa2, 0x80, 0x64, 0x80, 0xa0, 0x80, 0x62, 0x80, 0xa0, 0x03, 0x02,
				0x01, 0x20, 0xa1, 0x03, 0x02, 0x01, 0xfd, 0xdf, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xcf, 0xff, 0xfe, 0x00, 0x0e, 0x00, 0x01,
				0x40, 0x01, 0x02, 0x28, 0x02, 0x01, 0x23, 0xff,
			},
			false,
		},
		{
			"-pathErr",
			args{
				"node",
				"pathElement",
			},
			nil,
			true,
		},
		{
			"-unknownElementTypeErr",
			args{
				"foobar",
				"1",
			},
			nil,
			true,
		},
	}

	for _, tt := range tests {
		tt := tt
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			got, err := GetRequestByType(tt.args.et, tt.args.path)
			if (err != nil) != tt.wantErr {
				t.Fatalf("GetRequestByType() error = %v, wantErr %v", err, tt.wantErr)
			}

			if diff := cmp.Diff(tt.want, got); diff != "" {
				t.Fatalf("GetRequestByType() = %s", diff)
			}
		})
	}
}
