/*
** Copyright (C) 2001-2025 Zabbix SIA
**
** Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
** documentation files (the "Software"), to deal in the Software without restriction, including without limitation the
** rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
** permit persons to whom the Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in all copies or substantial portions
** of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
** WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
** COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
** TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
** SOFTWARE.
**/

package flag

import (
	"fmt"
	"os"
	"path/filepath"
	"runtime"
	"strings"

	"golang.zabbix.com/sdk/errs"
	"golang.zabbix.com/sdk/log"
	sdkplugin "golang.zabbix.com/sdk/plugin"
	"golang.zabbix.com/sdk/plugin/comms"
	"golang.zabbix.com/sdk/plugin/itemutil"
)

const usageMessageFormat = //
`%[1]s plugin for Zabbix agent 2

Usage of "%[2]s"
    %[2]s -h
    %[2]s -V
    %[2]s [-v] -t item-key

Options:
%[3]s
`

// Arguments holds the parsed command-line flags for the application.
type Arguments struct {
	test    string
	version bool
	help    bool
	verbose bool

	usageMessage string // received from the flags
}

type testFunction func(ctx sdkplugin.ContextProvider, key string, params []string, args ...any) (any, error)

// DecideActionFromFlags checks command-line flags and executes the corresponding action.
// Give pluginCustomTestFunction if plugin requires specific workflow, otherwise provide nil
// testArgs will be provided to that pluginCustomTestFunction if required.
func DecideActionFromFlags(
	args *Arguments,
	plugin any,
	pluginInfo *sdkplugin.Info,
	pluginCustomTestFunction testFunction,
	testArgs ...any,
) error {
	if args.verbose {
		err := setupVerboseLogging()
		if err != nil {
			return err
		}
	}

	switch {
	case args.version:
		return handleVersion(pluginInfo)
	case args.help:
		return handleHelp(pluginInfo, args.usageMessage)
	case args.test != "":
		return handleTest(args.test, plugin, pluginInfo, pluginCustomTestFunction, testArgs...)
	default:
		return nil
	}
}

func setupVerboseLogging() error {
	err := log.Open(log.Console, log.Trace, "", 0)
	if err != nil {
		return errs.Wrap(err, "failed to open log")
	}

	return nil
}

func handleVersion(pluginInfo *sdkplugin.Info) error {
	var sb strings.Builder

	// Write all version information to an in-memory strings.Builder to ignore errors
	fmt.Fprintf(&sb, "Zabbix %s plugin\n", pluginInfo.Name)
	fmt.Fprintf(&sb, "Version %d.%d.%d%s, built with %s\n",
		pluginInfo.MajorVersion,
		pluginInfo.MinorVersion,
		pluginInfo.PatchVersion,
		pluginInfo.Alphatag,
		runtime.Version(),
	)
	fmt.Fprintf(&sb, "Protocol version %s\n\n", comms.ProtocolVersion)
	fmt.Fprintln(&sb, pluginInfo.CopyrightMessage)

	// Single I/O operation
	_, err := fmt.Fprint(os.Stdout, sb.String())
	if err != nil {
		return errs.Wrap(err, "failed to write version info")
	}

	return errs.ErrExitGracefully
}

func handleHelp(pluginInfo *sdkplugin.Info, usageMessage string) error {
	_, err := fmt.Fprintf(
		os.Stdout,
		usageMessageFormat,
		pluginInfo.Name, filepath.Base(pluginInfo.BinName), usageMessage,
	)
	if err != nil {
		return errs.Wrap(err, "failed to write usage message")
	}

	return errs.ErrExitGracefully
}

func handleTest(
	testKey string,
	plugin any,
	pluginInfo *sdkplugin.Info,
	pluginCustomTestFunction testFunction,
	testArgs ...any,
) error {
	key, keyArgs, err := itemutil.ParseKey(testKey)
	if err != nil {
		return errs.Wrap(err, "failed to parse test key")
	}

	// if the plugin requires running with a custom function instead of the default way
	if pluginCustomTestFunction != nil {
		return executeCustomTest(pluginCustomTestFunction, key, keyArgs, testArgs...)
	}

	return executeDefaultTest(plugin, pluginInfo, key, keyArgs)
}

func executeCustomTest(testFunction testFunction, key string, keyArgs []string, testArgs ...any) error {
	response, err := testFunction(nil, key, keyArgs, testArgs)
	if err != nil {
		return errs.Wrap(err, "failed to execute test function")
	}

	_, err = fmt.Fprint(os.Stdout, response)
	if err != nil {
		return errs.Wrap(err, "failed to write response")
	}

	return errs.ErrExitGracefully
}

func executeDefaultTest(plugin any, pluginInfo *sdkplugin.Info, key string, keyArgs []string) error {
	accessor, ok := plugin.(sdkplugin.Accessor)
	if ok {
		accessor.Init(pluginInfo.Name)
	}

	configurator, ok := plugin.(sdkplugin.Configurator)
	if ok {
		configurator.Configure(&sdkplugin.GlobalOptions{Timeout: sdkplugin.DefaultPluginTimeout}, nil)
	}

	runner, ok := plugin.(sdkplugin.Runner)
	if ok {
		runner.Start()

		defer runner.Stop()
	}

	exporter, ok := plugin.(sdkplugin.Exporter)
	if ok {
		result, err := exporter.Export(key, keyArgs, nil)
		if err != nil {
			return errs.Wrap(err, "failed to export result")
		}

		_, err = fmt.Fprintln(os.Stdout, result)
		if err != nil {
			return errs.Wrap(err, "failed to write result")
		}

		return errs.ErrExitGracefully
	}

	return errs.New("plugin does not implement Export function, no data can be gathered")
}
