/*
** Copyright (C) 2001-2025 Zabbix SIA
**
** Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
** documentation files (the "Software"), to deal in the Software without restriction, including without limitation the
** rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
** permit persons to whom the Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in all copies or substantial portions
** of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
** WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
** COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
** TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
** SOFTWARE.
**/

package plugin

// revive:disable:max-public-structs

import (
	"sync/atomic"
	"time"

	"golang.zabbix.com/sdk/errs"
)

// DefaultPluginTimeout is default value for testing.
const DefaultPluginTimeout = 30 // seconds

// UnsupportedMetricError used to return that key is not supported.
//
//nolint:errname,gochecknoglobals //requires zabbix main repo fixes
var UnsupportedMetricError = errs.New("unsupported item key")

// Collector - interface for periodical metric collection.
type Collector interface {
	Collect() error
	Period() int
}

// Exporter - interface for exporting collected metrics.
type Exporter interface {
	// Export method exports data based on the key 'key' and its parameters 'params'.
	Export(
		key string, params []string, context ContextProvider,
	) (any, error)
}

// Runner - interface for managing background processes.
type Runner interface {
	// Start method activates plugin.
	Start()
	// Stop method deactivates plugin.
	Stop()
}

// Watcher - interface for fully custom monitoring.
type Watcher interface {
	// Watch method instructs plugin to watch for events based on item configuration.
	Watch(items []*Item, context ContextProvider)
}

// Configurator - interface for plugin configuration in agent conf files.
type Configurator interface {
	// Configure method passes global and private plugin configuration after it has been activated.
	Configure(globalOptions *GlobalOptions, privateOptions any)
	// Validate method validates private plugin configuration during agent startup.
	Validate(privateOptions any) error
}

// ResultWriter defines the contract for buffering and submitting collected check results.
type ResultWriter interface {
	Write(result *Result)
	Flush()
	SlotsAvailable() int
	PersistSlotsAvailable() int
}

// Meta holds thread-safe stateful metadata for an item that persists across checks.
type Meta struct {
	lastLogsize atomic.Uint64
	mtime       atomic.Int32
	Data        any
}

// ContextProvider provides the execution context and dependencies for a data collection task.
//
//nolint:interfacebloat
type ContextProvider interface {
	ClientID() uint64
	ItemID() uint64
	Output() ResultWriter
	Meta() *Meta
	GlobalRegexp() RegexpMatcher
	Timeout() int
	Delay() string
}

// Result encapsulates the data and metadata collected from a single item check.
type Result struct {
	Itemid uint64
	Value  *string

	// additional windows eventlog fields
	EventSource    *string
	EventID        *int
	EventTimestamp *int
	EventSeverity  *int

	Ts          time.Time
	Error       error
	LastLogsize *uint64
	Mtime       *int
	Persistent  bool
}

// Item represents a configured data collection task with its parameters.
type Item struct {
	Itemid      uint64
	Key         string
	Delay       string
	LastLogsize *uint64
	Mtime       *int
	Timeout     int
}

// GlobalOptions are global agent configuration parameters that can be accessed by plugins.
// In most cases it's recommended to allow plugins overriding global configuration parameters
// they are using with plugin specific parameters.
type GlobalOptions struct {
	Timeout  int    `json:"Timeout"`  //nolint:tagliatelle
	SourceIP string `json:"SourceIP"` //nolint:tagliatelle
}

// RegexpMatcher defines the interface for performing regular expression matches.
type RegexpMatcher interface {
	Match(value, pattern string, mode int, outputTemplate *string) (match bool, output string)
}

// SetLastLogsize sets lastLogSize.
func (m *Meta) SetLastLogsize(value uint64) {
	m.lastLogsize.Store(value)
}

// LastLogsize returns lastLogSize.
func (m *Meta) LastLogsize() uint64 {
	return m.lastLogsize.Load()
}

// SetMtime sets Mtime.
func (m *Meta) SetMtime(value int32) {
	m.mtime.Store(value)
}

// Mtime returns Mtime.
func (m *Meta) Mtime() int32 {
	return m.mtime.Load()
}
